package eu.dnetlib.validator2.validation.guideline.openaire;

import eu.dnetlib.validator2.engine.TestResultPredicate;
import eu.dnetlib.validator2.validation.guideline.Builders;
import eu.dnetlib.validator2.validation.guideline.ElementSpec;
import eu.dnetlib.validator2.validation.guideline.Guideline;
import eu.dnetlib.validator2.validation.guideline.SyntheticGuideline;
import eu.dnetlib.validator2.validation.utils.RegexValuePredicate;
import org.w3c.dom.Document;

import java.util.*;
import java.util.stream.Collectors;

import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE;
import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE_TO_N;
import static eu.dnetlib.validator2.validation.utils.SupportedRegExs.*;

public final class R1_2_01M_LIT_v3 extends AbstractOpenAireProfile {

    private static final String[] publicationVersions = {
            "info:eu-repo/semantics/draft",
            "info:eu-repo/semantics/submittedVersion",
            "info:eu-repo/semantics/acceptedVersion",
            "info:eu-repo/semantics/publishedVersion",
            "info:eu-repo/semantics/updateVersion"
    };

    private static final ElementSpec CREATOR_SPEC = Builders.
            forMandatoryElement("dc:creator", ONE_TO_N).
            build();

    private static final ElementSpec CONTRIBUTOR_SPEC = Builders.
            forMandatoryIfApplicableElement("datacite:contributor", ONE_TO_N, elementIsPresent("dc:contributor")).
            build();


    private static final ElementSpec PUBLICATION_DATE_SPEC = Builders.
            forMandatoryElement("dc:date", ONE).
            allowedValues(new RegexValuePredicate(COMPILED_PUBLICATION_DATE_REG_EX)).
            build();

    private static final ElementSpec RESOURCE_VERSION_SPEC = Builders.
            forRecommendedElement("dc:type").
            allowedValues(publicationVersions).
            build();

    private static final ElementSpec RELATED_IDENTIFIER_SPEC = Builders.
            forRecommendedRepeatableElement("dc:relation").
            allowedValues(relationSpecAllowedValuesPredicate()).
            build();

    private static TestResultPredicate<String> relationSpecAllowedValuesPredicate() {
        return new RegexValuePredicate(COMPILED_PROJECT_IDENTIFIER_REGEX).negate()
                .and(new RegexValuePredicate(COMPILED_ALT_IDENTIFIER_REG_EX).negate())
                .and(new RegexValuePredicate(COMPILED_PUBLICATION_REFERENCE_REG_EX).negate())
                .and(new RegexValuePredicate(COMPILED_DATASET_REFERENCE_REG_EX).negate());
    }

    //TODO: weights for guidelines haven't been finalized. They've been given an arbitrary value of 1.
    public static SyntheticGuideline R1_2_01M_LIT_1 = SyntheticGuideline.of("Creator", 4, CREATOR_SPEC);
    public static SyntheticGuideline R1_2_01M_LIT_2 = SyntheticGuideline.of("Contributor", 2, CONTRIBUTOR_SPEC);
    public static SyntheticGuideline R1_2_01M_LIT_3 = SyntheticGuideline.of("Date", 4, PUBLICATION_DATE_SPEC);
    public static SyntheticGuideline R1_2_01M_LIT_4 = SyntheticGuideline.of("Version", 2, RESOURCE_VERSION_SPEC);
    public static SyntheticGuideline R1_2_01M_LIT_5 = SyntheticGuideline.of("RelatedIdentifier", 2, RELATED_IDENTIFIER_SPEC);

    private static final List<Guideline<Document>> GUIDELINES = Collections.unmodifiableList(
            Arrays.asList(
                    R1_2_01M_LIT_1,
                    R1_2_01M_LIT_2,
                    R1_2_01M_LIT_3,
                    R1_2_01M_LIT_4,
                    R1_2_01M_LIT_5
            )
    );

    private static final Map<String, Guideline> GUIDELINE_MAP = GUIDELINES.
            stream().
            collect(Collectors.toMap(Guideline::getName, (guideline) -> guideline));

    private static final int MAX_SCORE = GUIDELINES.stream().map(Guideline::getWeight).reduce(0, Integer::sum);

    public R1_2_01M_LIT_v3() {
        super("(meta)data are associated with detailed provenance");
    }

    @Override
    public Collection<? extends Guideline<Document>> guidelines() {
        return GUIDELINES;
    }

    /**
     *
     * @param guidelineName
     * @return
     */
    @Override
    public Guideline guideline(String guidelineName) {
        return GUIDELINE_MAP.get(guidelineName);
    }

    @Override
    public int maxScore() {
        return MAX_SCORE;
    }
}