package eu.dnetlib.validator2.engine.builtins;

import eu.dnetlib.validator2.engine.RuleValidationResult;
import eu.dnetlib.validator2.engine.TestResultPredicate;
import eu.dnetlib.validator2.engine.contexts.TermsProperty;
import eu.dnetlib.validator2.engine.contexts.XMLCrisClassSchemeContextWithVocabulary;
import org.w3c.dom.NodeList;

import java.util.Map;

public class XMLCrisClassVocabularyRule extends XMLRule<XMLCrisClassSchemeContextWithVocabulary> {

    protected XMLCrisClassVocabularyRule(XMLCrisClassSchemeContextWithVocabulary context) {
        super(context, (NodeList nodes) -> {
            String schemeId = context.getCrisClassSchemeIdProperty().getValue();
            TermsProperty terms = context.getTermsProperty();
            String termsType = terms.getValue().toLowerCase();

            TestResultPredicate<String> termPredicate = (String value) -> {
                boolean exists = terms.termExists(value).isSuccess();
                if ( termsType.equals("blacklist") ) {
                    if ( exists )
                        return RuleValidationResult.failure("value '" + value + "' found in blacklist for scheme '" + schemeId + "'", "Remove the value or use a value that is not in the blacklist.");
                    else
                        return RuleValidationResult.success();
                } else { // whitelist
                    if ( exists )
                        return RuleValidationResult.success();
                    else
                        return RuleValidationResult.failure("value '" + value + "' not found in whitelist for scheme '" + schemeId + "'", "Use a value from the whitelist: " + terms.getValue());
                }
            };

            return context.getNodeListActionProperty().test(nodes, termPredicate);
        });
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder extends AbstractRuleBuilder<XMLCrisClassVocabularyRule, XMLCrisClassSchemeContextWithVocabulary> {

        Builder() {
            super(new StandardCrisClassXMLContextWithVocabulary());
            initialize();
        }

        private void initialize() {
            // Terms Type Property of VocabularyContext is not used here...
            context.getTermsTypeProperty().setValue("notNull");
        }

        public Builder setId(String id) {
            context.getIdProperty().setValue(id);
            return this;
        }

        public Builder setXPathExpression(String xpath) {
            context.getXPathExpressionProperty().setValue(xpath);
            return this;
        }

        public Builder setNodeListAction(String nodeListAction) throws RuntimeException {
            context.getNodeListActionProperty().setValue(nodeListAction);
            return this;
        }

        public Builder setVocabularyTerms(String terms) {
            context.getTermsProperty().setValue(terms);
            return this;
        }

        public Builder setCrisClassSchemeId(String id) {
            context.getCrisClassSchemeIdProperty().setValue(id);
            return this;
        }

        @Override
        public XMLCrisClassVocabularyRule build() {
            ensureContextIsValid();
            return new XMLCrisClassVocabularyRule(context);
        }

        @Override
        public XMLCrisClassVocabularyRule buildFrom(Map<String, String> map) {
            context.readFrom(map);
            ensureContextIsValid();
            return new XMLCrisClassVocabularyRule(context);
        }
    }

}
