package eu.dnetlib.validator2.validation.guideline;

import eu.dnetlib.validator2.engine.*;
import eu.dnetlib.validator2.engine.builtins.StandardRuleDiagnostics;
import eu.dnetlib.validator2.result_models.RequirementLevel;
import eu.dnetlib.validator2.result_models.StandardResult;
import eu.dnetlib.validator2.result_models.Status;
import eu.dnetlib.validator2.result_models.ValidationProblem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Evaluates a single, weighted guideline against a given XML document.
 * A guideline may be composed of multiple, interdependent rules. This class orchestrates the
 * evaluation of those rules and calculates a final score based on the guideline's weight and the
 * validation outcomes.
 */
class GuidelineEvaluation {

    private static final Logger logger = LoggerFactory.getLogger(GuidelineEvaluation.class);

    private static final boolean SHOULD_USE_OUT_ERR_DIAGNOSTICS = false;
    private static RuleDiagnostics<Document, Rule<Document>> OUT = null;
    private static RuleDiagnostics<Document, Rule<Document>> ERR = null;
    static {
        if ( SHOULD_USE_OUT_ERR_DIAGNOSTICS ) {
            OUT = Helper.Diagnostics.systemOut();
            ERR = Helper.Diagnostics.systemErr();
        }
    }

    private final String subjectId;
    private final Document doc;
    private final int weight;
    private final List<ValidationProblem> warnings = new ArrayList<>();
    private final List<ValidationProblem> errors = new ArrayList<>();
    private final Map<String, RequirementLevel> ruleIdToRequirementLevel = new HashMap<>();
    private final Map<String, Integer> ruleIdToNodeCount = new HashMap<>();
    private final Diagnostics diagnostics = new Diagnostics();
    private final Reporter<Document, SyntheticRule<Document>> reporter = new Reporter<>(diagnostics);

    /**
     * Initializes a new guideline evaluation.
     * @param subjectId A unique identifier for the validation subject (e.g., a file name).
     * @param doc The XML document to be validated.
     * @param weight The score assigned to this guideline if it passes successfully.
     */
    GuidelineEvaluation(String subjectId, Document doc, int weight) {
        this.subjectId = subjectId;
        this.doc = doc;
        this.weight = weight;
    }

    /**
     * Evaluates the rules defined in the CompilationResult against the document.
     *
     * @param result The compiled set of rules that constitute the guideline.
     * @return A StandardResult object containing the outcome of the evaluation. The score is determined
     *         by the initial weight, but is set to 0 if a RECOMMENDED or OPTIONAL rule is not populated,
     *         or if a MANDATORY rule's failure is demoted to a warning because its parent is not mandatory.
     */
    StandardResult evaluate(CompilationResult result)
    {
        boolean hasNotPopulatedWarningFromOptionalOrRecommendedOrNotApplicableRootElem = false;

        ruleIdToRequirementLevel.putAll(result.ruleIdToRequirementLevel);

        List<SyntheticRule<Document>> rules = new ArrayList<>();
        rules.add(result.rootNodeRule);
        rules.addAll(result.nodeRules);

        logger.debug("Evaluating " + this.subjectId + " with rules: " + rules);

        for ( SyntheticRule<Document> rule: rules )
        {
            RuleEngine.applyAndReport(rule, doc, reporter);

            Status status = diagnostics.getLastReportedStatus();
            if ( status == Status.ERROR )   // Fail fast in case of errors.
                return StandardResult.forError(diagnostics.getLastReportedError().getMessage());

            String id = rule.getContext().getIdProperty().getValue();
            RequirementLevel requirementLevel = getRequirementLevelOf(id);

            if ( status == Status.SUCCESS
                    && (requirementLevel == RequirementLevel.NOT_APPLICABLE) ) {
                // Report the non-applicability of a rule as a warning.
                // This is considered a "not populated" case for scoring purposes.
                logger.warn("Non-applicable: " + id);
                warnings.add(new ValidationProblem(synthesizeNotApplicableMessage(rule), null));
                hasNotPopulatedWarningFromOptionalOrRecommendedOrNotApplicableRootElem = true;
            }
            else if ( status == Status.FAILURE )
            {
                if ( requirementLevel == RequirementLevel.MANDATORY
                    || requirementLevel == RequirementLevel.MANDATORY_IF_APPLICABLE )
                {
                    // A mandatory rule has failed. We need to check its parent to decide if this is a true
                    // error or should be demoted to a warning.
                    SyntheticRule<Document> parentRule = rule.parentRule();
                    if ( parentRule == null ) {
                        // This is the root rule failing! This is a critical error.
                        logger.error("Important error for root failure: " + id + " -> " + diagnostics.lastFailureMessage);
                        errors.add(new ValidationProblem(synthesizeFailureMessage(rule), diagnostics.lastSuggestion));
                        // Here the zero-scoring already applies.
                    } else {
                        String parentRuleId = parentRule.getContext().getIdProperty().getValue();
                        RequirementLevel parentRequirementLevel = getRequirementLevelOf(parentRuleId);

                        if ( diagnostics.statusFor(parentRuleId) == Status.FAILURE )
                            hasNotPopulatedWarningFromOptionalOrRecommendedOrNotApplicableRootElem = true;

                        if ( parentRequirementLevel == RequirementLevel.MANDATORY
                                || parentRequirementLevel == RequirementLevel.MANDATORY_IF_APPLICABLE ) {
                            // The parent is also mandatory, so this failure is a critical error.
                            logger.error("Important error for main element failure: " + id + " -> " + diagnostics.lastFailureMessage);
                            errors.add(new ValidationProblem(synthesizeFailureMessage(rule), diagnostics.lastSuggestion));
                        } else {
                            // The parent is optional or recommended. The failure of this mandatory child rule
                            // should be reported as a warning, not an error.
                            if ( (diagnostics.statusFor(parentRuleId) == Status.FAILURE) && !isRulePopulated(parentRule, null, null) ) {
                                // The optional/recommended parent is not populated. We suppress the warning for the mandatory child
                                // to avoid redundant messages (e.g., warning about a missing child of a missing parent).
                                logger.info("Suppressing warning for mandatory rule with unpopulated optional/recommended parent: " + id);
                            } else {
                                // The parent is present (or failed for other reasons), so we report the child's failure as a warning.
                                // This is considered a "not populated" type of warning for scoring.
                                logger.warn("Mandatory " + (requirementLevel == RequirementLevel.MANDATORY_IF_APPLICABLE ? "_If_Applicable" : "") + " failure: " + id);
                                warnings.add(new ValidationProblem(synthesizeFailureMessage(rule), diagnostics.lastSuggestion));
                            }
                        }
                    }
                } else {
                    // The current rule is either "recommended" or "optional" (either parent opr sub-element/attribute).
                    if ( (requirementLevel == RequirementLevel.RECOMMENDED || requirementLevel == RequirementLevel.OPTIONAL)
                            && (rule.parentRule() == null)  // Zero the score only for root elements.
                            && !isRulePopulated(rule, null, null) ) {
                        // If an optional or recommended rule fails because it's not populated, trigger the zero-score rule.
                        hasNotPopulatedWarningFromOptionalOrRecommendedOrNotApplicableRootElem = true;
                    }

                    if ( requirementLevel == RequirementLevel.RECOMMENDED )
                        logger.warn("Recommended failure: " + id);
                    else if ( requirementLevel == RequirementLevel.OPTIONAL )
                        logger.warn("Optional failure: " + id);
                    else    // This should never happen, but better catch this case.
                        logger.error("UNKNOWN failure: " + id);

                    warnings.add(new ValidationProblem(synthesizeFailureMessage(rule), diagnostics.lastSuggestion));
                }
            }
            // Else it is just "SUCCESS" for an applicable rule, which means there is no error or warning to create.
        }

        if ( !errors.isEmpty() )
            return StandardResult.forFailure(warnings, errors);
        else
            return StandardResult.forSuccess(hasNotPopulatedWarningFromOptionalOrRecommendedOrNotApplicableRootElem ? 0 : weight, warnings);
    }

    /**
     * Checks if the element or attribute targeted by a rule exists in the document.
     * This is used to differentiate between a rule failing because a field is missing versus
     * failing because the field's content is invalid.
     *
     * @param rule The rule to check.
     * @param nodeType The type of node ('element' or 'attribute'). If null, it's parsed from the rule ID.
     * @param elementOrAttribute The name of the element or attribute. If null, it's parsed from the rule ID.
     * @return True if the targeted node is found, false otherwise.
     */
    private boolean isRulePopulated(SyntheticRule<Document> rule, String nodeType, String elementOrAttribute)
    {
        Matcher matcher = null;
        if ( nodeType == null ) {
            String ruleId = rule.getContext().getIdProperty().getValue();
            matcher = RULE_PATTERN.matcher(ruleId);
            if ( !matcher.find() )
                return false;
            nodeType = matcher.group(4);
            if ( nodeType == null )
                return false;
            nodeType = nodeType.trim();
        }

        if ( "element".equals(nodeType) ) {
            NodeList testedNodes = rule.getTestedNodes();
            return (testedNodes != null && testedNodes.getLength() > 0);
        } else if ( "attribute".equals(nodeType) ) {
            SyntheticRule<Document> parentRule = rule.parentRule();
            if ( parentRule != null ) {
                NodeList parentNodes = parentRule.getTestedNodes();
                int numParentNodes = (parentNodes != null) ? parentNodes.getLength() : 0;
                if ( numParentNodes > 0 ) {
                    if ( (elementOrAttribute == null) && (elementOrAttribute = matcher.group(3)) == null )
                        return false;
                    String[] pathParts = elementOrAttribute.split("/");
                    String attrName = pathParts[pathParts.length - 1];
                    if ( attrName.isEmpty() )
                        return false;

                    for ( int i = 0; i < numParentNodes; i++ ) {
                        Node parentNode = parentNodes.item(i);
                        if ( parentNode.getNodeType() == Node.ELEMENT_NODE && ((Element)parentNode).hasAttribute(attrName) )
                            return true;
                    }
                }
            }
        }
        return false;
    }


    /**
     * This pattern is used to parse user-friendly information from a rule's unique ID string.
     * The ID contains structured information about the rule's target, cardinality, and type.
     * Example ID: "The title rule: 'dc:title' element, has cardinality one"
     * Groups:
     * 1: Introduction (e.g., "The title rule: '")
     * 2: Parent element context (optional, e.g., "dc:description/")
     * 3: Element/Attribute name (e.g., "xml:lang")
     * 4: Node type (e.g., " element" or " attribute")
     * 5: Cardinality (e.g., "one-to-n")
     */
    private static final Pattern RULE_PATTERN = Pattern.compile("(The [\\w_]+ rule: ')(?:([\\w:/]+)/)?([\\w:\\s]+)'(\\s(?:element|attribute))[\\w\\s]*,? has cardinality (\\w+(?:-\\w+)?)");

    /**
     * Constructs a user-friendly message by parsing the rule's ID.
     * This avoids having to pass around separate descriptive fields for each rule.
     *
     * @param rule The rule to generate a message for.
     * @param failureMessage The specific reason for failure, if any.
     * @return A synthesized, human-readable message describing the rule's requirement.
     */
    private String synthesizeMainMessage(Rule<Document> rule, String failureMessage)
    {
        String ruleId = rule.getContext().getIdProperty().getValue();
        Matcher matcher = RULE_PATTERN.matcher(ruleId);
        if ( ! matcher.find() ) {
            logger.error("Unknown rule ID detected. Falling back to simple message for: " + ruleId);
            return ruleId + ", has failed";
        }
        String introduction = matcher.group(1);
        if ( (introduction == null) || (introduction = introduction.trim()).isEmpty() ) {
            logger.error("Could not get the 'introduction' from ruleId: " + ruleId);
            return ruleId + ", has failed";
        }
        String parentElement = matcher.group(2);  // parentElement is optional.
        if ( (parentElement != null) && (parentElement = parentElement.trim()).isEmpty() )
            parentElement = null;
        String elementOrAttribute = matcher.group(3);
        if ( (elementOrAttribute == null) || (elementOrAttribute = elementOrAttribute.trim()).isEmpty() ) {
            logger.error("Could not get the 'elementOrAttribute' from ruleId: " + ruleId);
            return ruleId + ", has failed";
        }
        // The regex for group 4 intentionally captures a leading space (e.g., " element").
        // This is required for correct message formatting and should not be trimmed.
        String nodeType = matcher.group(4);
        if ( (nodeType == null) || nodeType.isEmpty() ) {
            logger.error("Could not get the 'nodeType' from ruleId: " + ruleId);
            return ruleId + ", has failed";
        }
        String cardinality = matcher.group(5);
        if ( (cardinality == null) || (cardinality = cardinality.trim()).isEmpty() ) {
            logger.error("Could not get the 'cardinality' from ruleId: " + ruleId);
            return ruleId + ", has failed";
        }

        RequirementLevel ruleRequirementLevel = getRequirementLevelOf(ruleId);
        String lastMessage;

        if ( ruleRequirementLevel == RequirementLevel.OPTIONAL || ruleRequirementLevel == RequirementLevel.RECOMMENDED )
            lastMessage = ((rule instanceof SyntheticRule) && isRulePopulated((SyntheticRule<Document>) rule, nodeType, elementOrAttribute))
                    ? ", has failed" : ", is not populated";
        else { // Mandatory or other levels
            if ( (failureMessage != null) && failureMessage.contains("but found 0") )
                lastMessage = ", is not populated";
            else
                lastMessage = ", has failed";
        }

        final StringBuilder sb = new StringBuilder(100);
        sb.append(introduction).append(elementOrAttribute).append("'").append(nodeType.toLowerCase());
        if ( parentElement != null ) {
            sb.append(" of the '").append(parentElement).append("' element");
        }
        sb.append(", with expected occurrence ").append(cardinality).append(lastMessage);
        return sb.toString();
    }


    private String synthesizeFailureMessage(Rule<Document> rule)
    {
        String failureMessage = diagnostics.lastFailureMessage;
        
        // Adjust failure message for optional/recommended rules to match the "0-n" or "0-1" expectation
        // displayed in the main message.
        String ruleId = rule.getContext().getIdProperty().getValue();
        RequirementLevel level = getRequirementLevelOf(ruleId);
        if ( level == RequirementLevel.OPTIONAL || level == RequirementLevel.RECOMMENDED) {
             if ( failureMessage.contains("expected cardinality 1-n") )
                 failureMessage = failureMessage.replace("expected cardinality 1-n", "expected cardinality 0-n");
             else if ( failureMessage.contains("expected cardinality 1") )
                 failureMessage = failureMessage.replace("expected cardinality 1", "expected cardinality 0-1");
        }

        String message = synthesizeMainMessage(rule, failureMessage);
        return message + " (reason: '" + failureMessage + "').";
    }


    private String synthesizeNotApplicableMessage(Rule<Document> rule) {
        String message = synthesizeMainMessage(rule, null);
        return message + ", is not applicable.";
    }


    void setNodeCountOf(String ruleId, int count) {
        ruleIdToNodeCount.put(ruleId, count);
    }

    Integer getNodeCountOf(String ruleId) {
        return ruleIdToNodeCount.get(ruleId);
    }

    void setRequirementLevelOf(String ruleId, RequirementLevel requirementLevel) {
        ruleIdToRequirementLevel.put(ruleId, requirementLevel);
    }

    RequirementLevel getRequirementLevelOf(String ruleId) {
        return ruleIdToRequirementLevel.get(ruleId);
    }

    @Override
    public String toString() {
        return "GuidelineEvaluation{" +
                "subjectId='" + subjectId + '\'' +
                ", doc=" + doc +
                ", weight=" + weight +
                ", warnings=" + warnings +
                ", errors=" + errors +
                ", ruleIdToRequirementLevel=" + ruleIdToRequirementLevel +
                ", ruleIdToNodeCount=" + ruleIdToNodeCount +
                ", diagnostics=" + diagnostics +
                ", reporter=" + reporter +
                '}';
    }


    private static final class Diagnostics extends StandardRuleDiagnostics<Document, SyntheticRule<Document>>
    {
        private final Map<String, Status> statusByRuleId = new HashMap<>();
        private String lastFailureMessage;
        private String lastSuggestion;

        @Override
        public void success(SyntheticRule<Document> rule, Document document) {
            if ( SHOULD_USE_OUT_ERR_DIAGNOSTICS )
                OUT.success(rule, document);
            super.success(rule, document);
            statusByRuleId.put(rule.getContext().getIdProperty().getValue(), Status.SUCCESS);
            this.lastFailureMessage = null; // Clear last failure-message upon reaching a successful rule.
            this.lastSuggestion = null;
        }

        @Override
        public void failure(SyntheticRule<Document> rule, Document document, String message, String suggestion) {
            if ( SHOULD_USE_OUT_ERR_DIAGNOSTICS )
                OUT.failure(rule, document, message, suggestion);
            super.failure(rule, document, message, suggestion);
            this.lastFailureMessage = message;
            this.lastSuggestion = suggestion;
            statusByRuleId.put(rule.getContext().getIdProperty().getValue(), Status.FAILURE);
        }

        public String getLastFailureMessage() {
            return lastFailureMessage;
        }

        public String getLastSuggestion() {
            return lastSuggestion;
        }

        @Override
        public void error(SyntheticRule<Document> rule, Document document, Throwable err) {
            if ( SHOULD_USE_OUT_ERR_DIAGNOSTICS )
                ERR.error(rule, document, err);
            super.error(rule, document, err);
            statusByRuleId.put(rule.getContext().getIdProperty().getValue(), Status.ERROR);
        }

        public Status statusFor(String ruleId) {
            return statusByRuleId.get(ruleId);
        }

        @Override
        public String toString() {
            return "Diagnostics{" +
                    "statusByRuleId=" + statusByRuleId +
                    ", lastFailureMessage='" + lastFailureMessage + '\'' +
                    ", lastSuggestion='" + lastSuggestion + '\'' +
                    '}';
        }
    }
}
