package eu.dnetlib.broker.openaireAlerts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.broker.LiteratureBrokerServiceApplication;
import eu.dnetlib.broker.common.elasticsearch.AlertNotification;
import eu.dnetlib.broker.common.elasticsearch.AlertNotificationRepository;
import eu.dnetlib.broker.common.elasticsearch.Notification;
import eu.dnetlib.broker.common.properties.ElasticSearchProperties;
import eu.dnetlib.broker.common.subscriptions.ConditionOperator;
import eu.dnetlib.broker.common.subscriptions.ConditionParams;
import eu.dnetlib.broker.common.subscriptions.MapCondition;
import eu.dnetlib.broker.common.subscriptions.NotificationFrequency;
import eu.dnetlib.broker.common.subscriptions.NotificationMode;
import eu.dnetlib.broker.common.subscriptions.Subscription;
import eu.dnetlib.broker.common.subscriptions.SubscriptionRepository;
import eu.dnetlib.broker.common.utils.MapValueType;
import eu.dnetlib.broker.events.output.DispatcherManager;
import eu.dnetlib.broker.objects.alerts.ValidatorAlertMessage;
import eu.dnetlib.common.controller.AbstractDnetController;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Profile("openaire")
@RestController
@RequestMapping("/api/openaire-alerts")
@Tag(name = LiteratureBrokerServiceApplication.TAG_OPENAIRE_ALERTS)
public class OpenaireAlertsBrokerController extends AbstractDnetController {

	@Autowired
	private ElasticsearchOperations esOperations;

	@Autowired
	private ElasticSearchProperties props;

	@Autowired
	private SubscriptionRepository subscriptionRepo;

	@Autowired
	private JdbcTemplate jdbcTemplate;

	@Autowired
	private DispatcherManager dispatcher;

	@Autowired
	private AlertNotificationRepository alertNotificationRepository;

	private static final Log log = LogFactory.getLog(OpenaireAlertsBrokerController.class);

	@Operation(summary = "Return the datasources having alerts")
	@GetMapping("/datasources-with-alerts")
	public List<DatasourceWithAlert> findDatasourcesWithAlerts() {
		try {
			final String sql = IOUtils.toString(getClass().getResourceAsStream("/sql/datasourceWithAlerts.sql"), "UTF-8");
			final RowMapper<DatasourceWithAlert> mapper =
					(rs, rowNum) -> new DatasourceWithAlert(rs.getString("id"), rs.getString("name"), rs.getString("rule"), rs.getLong("size"));
			return jdbcTemplate.query(sql, mapper);
		} catch (final Exception e) {
			log.error("Error executing query", e);
			return new ArrayList<>();
		}
	}

	@Operation(summary = "Perform a subscription")
	@PostMapping("/subscribe/{compatibility}")
	public Subscription registerSubscription(@PathVariable final String compatibility, @RequestParam final String email, @RequestParam final String dsId) {
		if (StringUtils.isBlank(email)) { throw new IllegalArgumentException("subscriber is empty"); }

		final String topic = "ALERT/" + compatibility;

		for (final Subscription s : subscriptionRepo.findBySubscriber(email)) {
			if (topic.equalsIgnoreCase(s.getTopic()) && dsId.equalsIgnoreCase(extractDatasourceId(s))) {
				throw new IllegalArgumentException("Already subscribed");
			}
		}

		final String subscriptionId = "sub-" + UUID.randomUUID();

		final List<MapCondition> conds = new ArrayList<>();
		conds.add(new MapCondition("datasourceId", MapValueType.STRING, ConditionOperator.EXACT, Arrays.asList(new ConditionParams(dsId, null))));

		final Subscription s =
				new Subscription(subscriptionId, email, topic, NotificationFrequency.realtime, NotificationMode.EMAIL, null, new Date(),
						conds);

		return subscriptionRepo.save(s);
	}

	@Operation(summary = "Return the subscriptions of an user (by email and datasource (optional))")
	@GetMapping("/subscriptions")
	public List<AlertSubscriptionDesc> subscriptions(@RequestParam final String email) {
		final Iterable<Subscription> iter = subscriptionRepo.findBySubscriber(email);

		return StreamSupport.stream(iter.spliterator(), false)
				.filter(s -> s.getTopic().startsWith("ALERT/"))
				.map(this::subscriptionDesc)
				.sorted(Comparator.comparing(AlertSubscriptionDesc::getDsName))
				.collect(Collectors.toList());
	}

	private AlertSubscriptionDesc subscriptionDesc(final Subscription s) {
		final String dsId = extractDatasourceId(s);
		final String dsName = findDatasouceName(dsId);

		return new AlertSubscriptionDesc(s.getSubscriptionId(), dsId, dsName, s.getTopic(), s.getCreationDate(), s.getLastNotificationDate(),
				alertNotificationRepository.countBySubscriptionId(s.getSubscriptionId()));
	}

	private String findDatasouceName(final String dsId) {
		return StringUtils.firstNonBlank(jdbcTemplate
				.queryForObject("select name from oa_datasource_stats where id=? and topic ilike 'ALERT/%' limit 1", String.class, dsId), dsId);
	}

	private String extractDatasourceId(final Subscription sub) {
		return sub.getConditionsAsList()
				.stream()
				.filter(c -> "datasourceId".equals(c.getField()))
				.map(MapCondition::getListParams)
				.filter(l -> !l.isEmpty())
				.map(l -> l.get(0).getValue())
				.findFirst()
				.orElse("");
	}

	@Operation(summary = "Return a page of alert notifications")
	@GetMapping("/notifications/{subscrId}/{nPage}/{size}")
	public AlertsPage notifications(@PathVariable final String subscrId, @PathVariable final int nPage, @PathVariable final int size) {
		final Optional<Subscription> optSub = subscriptionRepo.findById(subscrId);

		if (optSub.isPresent()) {
			final Subscription sub = optSub.get();

			final NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
					.withQuery(QueryBuilders.termQuery("subscriptionId.keyword", subscrId))
					.withSearchType(SearchType.DEFAULT)
					.withFields("payload")
					.withPageable(PageRequest.of(nPage, size))
					.build();

			final SearchHits<AlertNotification> page =
					esOperations.search(searchQuery, AlertNotification.class, IndexCoordinates.of(props.getAlertNotificationsIndexName()));

			final List<ValidatorAlertMessage> list = page.stream()
					.map(SearchHit::getContent)
					.map(Notification::getPayload)
					.map(ValidatorAlertMessage::fromJSON)
					.collect(Collectors.toList());

			return new AlertsPage(extractDatasourceId(sub), sub.getTopic(), nPage, overrideGetTotalPage(page, size), page.getTotalHits(), list);
		}
		log.warn("Invalid subscription: " + subscrId);
		return new AlertsPage("", "", nPage, 0, 0, new ArrayList<>());
	}

	@Operation(summary = "Send notifications")
	@GetMapping("/notifications/sendMailForDatasource")
	private List<String> sendMailForNotifications(@RequestParam final String dsId) {
		new Thread(() -> innerSendMailForNotifications(dsId)).start();
		return Arrays.asList("Sending ...");
	}

	private List<String> innerSendMailForNotifications(final String dsId) {
		for (final Subscription s : subscriptionRepo.findAll()) {
			if (s.getTopic().startsWith("ALERT/") && s.getConditionsAsList()
					.stream()
					.anyMatch(c -> "datasourcId".equals(c.getField()) && c.getListParams().stream().anyMatch(p -> dsId.equals(p.getValue())))) {

				final long count = alertNotificationRepository.countBySubscriptionId(s.getSubscriptionId());
				if (count > 0) {
					final Map<String, Object> params = new HashMap<>();
					params.put("oa_notifications_total", count);
					params.put("oa_datasource", extractDatasourceId(s));
					dispatcher.sendNotification(s, params);
				}

				s.setLastNotificationDate(new Date());
				subscriptionRepo.save(s);
			}
		}
		return Arrays.asList("Sending ...");
	}

	@Operation(summary = "Update stats")
	@GetMapping("/stats/update")
	public List<String> updateStats(@RequestParam final String dsId) {

		new Thread(() -> _updateStats(dsId)).start();
		return Arrays.asList("Sending ...");
	}

	protected void _updateStats(final String dsId) {
		try {
			final String tempTable = "oa_datasource_stats_temp_" + DigestUtils.md5Hex(dsId);
			final String sql = IOUtils.toString(getClass().getResourceAsStream("/sql/updateAlertStats.sql"), "UTF-8")
					.replaceAll("__TEMP_TABLE__", tempTable);
			jdbcTemplate.update(sql, dsId);
		} catch (final Exception e) {
			log.error("Error updating stats", e);
		}
	}

	private long overrideGetTotalPage(final SearchHits<?> page, final int size) {
		return (page.getTotalHits() + size - 1) / size;
	}
}
