package eu.dnetlib.data.collector.plugins.gtr2;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.function.Function;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.joda.time.DateTime;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.CollectorServiceRuntimeException;

public abstract class Gtr2Iterator implements Iterator<String> {

	public static final int PAGE_SIZE = 20;

	private static final Log log = LogFactory.getLog(Gtr2Iterator.class);

	private final String baseUrl;
	private int currPage;
	private int endPage;
	private boolean incremental = false;
	private DateTime fromDate;

	private final Map<String, String> cache = new HashMap<>();

	private final Queue<String> queue = new LinkedList<>();

	private String nextElement;

	public Gtr2Iterator(final String baseUrl, final String fromDate, final String startPage, final String endPage)
		throws CollectorServiceException {

		this.baseUrl = baseUrl;
		this.currPage = NumberUtils.toInt(startPage, 1);
		this.endPage = NumberUtils.toInt(endPage, Integer.MAX_VALUE);
		this.incremental = StringUtils.isNotBlank(fromDate);

		if (this.incremental) {
			this.fromDate = Gtr2Helper.parseDate(fromDate);
		}

		prepareNextElement();
	}

	@Override
	public boolean hasNext() {
		return nextElement != null;
	}

	@Override
	public String next() {
		try {
			return nextElement;
		} finally {
			prepareNextElement();
		}
	}

	@Override
	public void remove() {
		throw new UnsupportedOperationException();
	}

	private void prepareNextElement() {
		while (this.currPage <= this.endPage && queue.isEmpty()) {
			log.debug("FETCHING PAGE + " + currPage + "/" + endPage);
			this.queue.addAll(fetchPage(currPage++));
		}
		this.nextElement = this.queue.poll();
	}

	private List<String> fetchPage(final int pageNumber) {

		final List<String> res = new ArrayList<>();
		try {
			final Document doc = Gtr2Helper.loadURL(urlForPage(baseUrl, pageNumber)).getDocument();

			if (endPage == Integer.MAX_VALUE) {
				endPage = NumberUtils.toInt(doc.valueOf("/*/@*[local-name() = 'totalPages']"));
				log.debug("Changed number of total pages " +  endPage);
			}

			for (final Object po : doc.selectNodes(xpathForEntity())) {
				final Element mainEntity = (Element) ((Element) po).detach();

				if (filterIncremental(mainEntity)) {
					res.add(expandMainEntity(mainEntity));
				} else {
					log.debug("Skipped entity");
				}

			}
		} catch (final Throwable e) {
			log.error("Exception fetching page " + pageNumber, e);
			throw new CollectorServiceRuntimeException("Exception fetching page " + pageNumber, e);
		}

		return res;
	}

	protected void addLinkedEntities(final Element master, final String relType, final Element newRoot, final Function<Document, Element> mapper) {

		for (final Object o : master.selectNodes(".//*[local-name()='link']")) {
			final String rel = ((Element) o).valueOf("@*[local-name()='rel']");
			final String href = ((Element) o).valueOf("@*[local-name()='href']");

			if (relType.equals(rel) && StringUtils.isNotBlank(href)) {
				final String cacheKey = relType + "#" + href;
				if (cache.containsKey(cacheKey)) {
					try {
						log.debug(" * from cache (" + relType + "): " + href);
						newRoot.add(DocumentHelper.parseText(cache.get(cacheKey)).getRootElement());
					} catch (final DocumentException e) {
						log.error("Error retrieving cache element: " + cacheKey, e);
						throw new CollectorServiceRuntimeException("Error retrieving cache element: " + cacheKey, e);
					}
				} else {
					try {
						final Document doc = Gtr2Helper.loadURL(href).getDocument();
						final Element elem = mapper.apply(doc);
						newRoot.add(elem);
						cache.put(cacheKey, elem.asXML());
					} catch (final Throwable e) {
						log.warn("Invalid url: " + href);
					}
				}

			}
		}
	}

	private boolean filterIncremental(final Element e) {
		if (!incremental) {
			return true;
		} else if (Gtr2Helper.isAfter(e.valueOf("@*[local-name() = 'created']"), fromDate)) {
			return true;
		} else if (Gtr2Helper.isAfter(e.valueOf("@*[local-name() = 'updated']"), fromDate)) {
			return true;
		} else {
			return false;
		}
	}

	abstract protected String expandMainEntity(final Element mainEntity);

	abstract protected String urlForPage(final String baseUrl, final int pageNumber);

	abstract protected String xpathForEntity();
}
