package eu.dnetlib.data.collector.plugins.eosc;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONObject;

import com.google.common.collect.Iterators;

import eu.dnetlib.data.collector.plugin.AbstractCollectorPlugin;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.InterfaceDescriptor;

public class EoscServicesPlugin extends AbstractCollectorPlugin {

	// Suggested values:
	// baseUrl = https://api.eosc-portal.eu/public
	// XPATH_ID = /results/id
	// Go to https://aai.eosc-portal.eu/providers-api/ (user: d-net) to see the refreshToken and the clientId

	private static final Log log = LogFactory.getLog(EoscServicesPlugin.class);

	@Override
	public Iterable<String> collect(final InterfaceDescriptor interfaceDescriptor, final String fromDate, final String untilDate)
		throws CollectorServiceException {

		final String baseUrl = interfaceDescriptor.getBaseUrl();

		final long quantity = NumberUtils.toLong(interfaceDescriptor.getParams().get("quantity"), 10000);
		final String type = interfaceDescriptor.getParams().get("type");

		final String authorizationUrl = interfaceDescriptor.getParams().get("authorizationUrl");
		final String clientId = interfaceDescriptor.getParams().get("clientId");
		final String refreshToken = interfaceDescriptor.getParams().get("refreshToken");

		final String accessToken = obtainAccessoToken(authorizationUrl, clientId, refreshToken);

		final String urlOrgs = baseUrl + "/provider/bundle/all?quantity=" + quantity;
		String urlServices = baseUrl + "/service/adminPage/all?quantity=" + quantity;
		if (StringUtils.isNotBlank(type)) {
			urlServices += "&type=" + type.trim();
		}

		final Iterator<String> iter1 = iterCall(urlOrgs, accessToken, "organization");
		final Iterator<String> iter2 = iterCall(urlServices, accessToken, "service");

		return () -> Iterators.concat(iter1, iter2);
	}

	private String obtainAccessoToken(final String url, final String clientId, final String refreshToken) throws CollectorServiceException {
		try (final CloseableHttpClient client = HttpClients.createDefault()) {
			final HttpPost req = new HttpPost(url);
			final List<NameValuePair> params = new ArrayList<>();
			params.add(new BasicNameValuePair("grant_type", "refresh_token"));
			params.add(new BasicNameValuePair("refresh_token", refreshToken));
			params.add(new BasicNameValuePair("client_id", clientId));
			params.add(new BasicNameValuePair("scope", "openid email profile"));

			req.setEntity(new UrlEncodedFormEntity(params, "UTF-8"));

			try (final CloseableHttpResponse response = client.execute(req)) {
				final String content = IOUtils.toString(response.getEntity().getContent());
				final JSONObject obj = new JSONObject(content);
				final String token = obj.getString("access_token");
				if (StringUtils.isNotBlank(token)) { return token; }
			}
		} catch (final Throwable e) {
			log.warn("Error obtaining access token", e);
			throw new CollectorServiceException("Error obtaining access token", e);
		}
		throw new CollectorServiceException("Access token is missing");
	}

	@SuppressWarnings("unchecked")
	private Iterator<String> iterCall(final String url, final String token, final String type) {
		return httpCall(url, token)
			.selectNodes("//results/results")
			.stream()
			.map(o -> {
				final Element el = (Element) o;
				el.addAttribute("type", type);
				return el.asXML();
			})
			.iterator();
	}

	private Document httpCall(final String url, final String token) {
		log.info("EOSC API: " + url);

		final SAXReader reader = new SAXReader();

		final HttpGet req = new HttpGet(url);
		req.addHeader("Content-Type", "application/xml");
		req.addHeader("Accept", "application/xml");
		if (StringUtils.isNotBlank(token)) {
			req.addHeader("Authorization", "Bearer " + token);
		}

		try (final CloseableHttpClient client = HttpClients.createDefault()) {
			try (final CloseableHttpResponse response = client.execute(req)) {
				return reader.read(response.getEntity().getContent());
			}
		} catch (final Exception e) {
			throw new RuntimeException(e);
		}
	}

}
