package eu.dnetlib.msro.workflows.dedup;

import java.util.Queue;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.dedup.conf.DedupConfigurationOrchestration;
import eu.dnetlib.msro.workflows.dedup.conf.DedupConfigurationOrchestrationLoader;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;

public class DedupCheckEntitySequenceJobNode extends AsyncJobNode {

	private static final Log log = LogFactory.getLog(DedupCheckEntitySequenceJobNode.class);

	@Autowired
	private DedupConfigurationOrchestrationLoader dedupOrchestrationLoader;

	private String dedupConfigSequenceParam;

	private String entitySequence;

	@Override
	protected String execute(final NodeToken token) throws Exception {

		if (StringUtils.isBlank(getEntitySequence())) throw new MSROException("missing entity sequence, e.g. a csv: organization,person,result");

		if (token.getFullEnv().hasAttribute(DedupGrouperJobNode.DEDUP_GROUPER_LOOPER)) {
			log.info("reset env variable: " + DedupGrouperJobNode.DEDUP_GROUPER_LOOPER + " to zero");
			token.getFullEnv().setAttribute(DedupGrouperJobNode.DEDUP_GROUPER_LOOPER, 0);
		}

		if (!token.getEnv().hasAttribute("entitySequence")) {

			log.info("parsing config sequence: " + getEntitySequence());

			token.getEnv().setAttribute("entitySequence", getEntitySequence());

			final Iterable<String> sequence = Splitter.on(",").omitEmptyStrings().split(getEntitySequence());
			final Queue<DedupConfigurationOrchestration> q =
					Lists.newLinkedList(Iterables.transform(sequence, new Function<String, DedupConfigurationOrchestration>() {

						@Override
						public DedupConfigurationOrchestration apply(final String entityName) {
							try {
								final DedupConfigurationOrchestration dco = Iterables.getFirst(dedupOrchestrationLoader.loadByEntityName(entityName), null);
								if (dco == null) throw new RuntimeException("unable to find DedupOrchestration profile for entity type: " + entityName);
								return dco;
							} catch (final Throwable e) {
								throw new RuntimeException("", e);
							}
						}
					}));

			log.info("built sequence of dedup orchestration profiles, size: " + q.size());
			final DedupConfigurationOrchestration dco = q.remove();
			log.info("closing mesh for entity: " + dco.getEntity().getName());
			setDedupConfParams(token, dco);
			token.getEnv().setTransientAttribute("entitySequenceQueue", q);

			return Arc.DEFAULT_ARC;
		}

		@SuppressWarnings("unchecked")
		final Queue<DedupConfigurationOrchestration> q = (Queue<DedupConfigurationOrchestration>) token.getEnv().getTransientAttribute("entitySequenceQueue");

		if (!q.isEmpty()) {
			log.info("remaining dedup orchestration profiles: " + q.size());
			final DedupConfigurationOrchestration dco = q.remove();
			log.info("closing mesh for entity: " + dco.getEntity().getName());

			setDedupConfParams(token, dco);
			return Arc.DEFAULT_ARC;
		}

		log.info("completed closing mesh for entities: " + getEntitySequence());
		return "done";

	}

	private void setDedupConfParams(final NodeToken token, final DedupConfigurationOrchestration dco) {
		token.getEnv().setAttribute("entityType", dco.getEntity().getName());
		token.getEnv().setAttribute("entityTypeId", dco.getEntity().getCode());
		token.getEnv().setAttribute(getDedupConfigSequenceParam(), dco.toString());
	}

	public String getEntitySequence() {
		return entitySequence;
	}

	public void setEntitySequence(final String entitySequence) {
		this.entitySequence = entitySequence;
	}

	public String getDedupConfigSequenceParam() {
		return dedupConfigSequenceParam;
	}

	public void setDedupConfigSequenceParam(final String dedupConfigSequenceParam) {
		this.dedupConfigSequenceParam = dedupConfigSequenceParam;
	}

}
