package eu.dnetlib.msro.workflows.actions;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.actionmanager.set.RawSet;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.dedup.conf.DedupConfigurationOrchestration;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * The Class PrepareConfiguredActionSetJobNode.
 */
public class PrepareConfiguredActionSetJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareConfiguredActionSetJobNode.class);

	/**
	 * The dedup config sequence param.
	 */
	private String dedupConfigSequenceParam;

	/**
	 * The job property.
	 */
	private String jobProperty;

	/**
	 * The action set path param name.
	 */
	private String actionSetPathParam;

	/**
	 * The service locator.
	 */
	@Autowired
	private UniqueServiceLocator serviceLocator;

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.msro.workflows.nodes.SimpleJobNode#execute(com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected String execute(final NodeToken token) throws Exception {

		final List<Map<String, String>> setList = Lists.newArrayList();

		final Map<String, String> set = Maps.newHashMap();

		final String actionSetId = getActionSetId(token);
		final ISLookUpService isLookUpService = serviceLocator.getService(ISLookUpService.class);
		final String basePath = isLookUpService.getResourceProfileByQuery(
				"/RESOURCE_PROFILE[./HEADER/RESOURCE_TYPE/@value='ActionManagerServiceResourceType']//SERVICE_PROPERTIES/PROPERTY[@key='basePath']/@value/string()");
		if (StringUtils.isBlank(basePath)) {
			throw new IllegalStateException("missing basePath in ActionManagerService");
		}

		final String actionSetDirectory = isLookUpService.getResourceProfileByQuery(
				"/RESOURCE_PROFILE[./HEADER/RESOURCE_TYPE/@value='ActionManagerSetDSResourceType' and .//SET/@id = '"+actionSetId+"']//SET/@ directory/string()");

		if (StringUtils.isBlank(actionSetDirectory)) {
			throw new IllegalStateException("missing directory in ActionSet profile: " + actionSetId);
		}

		final String rawSetId = RawSet.newInstance().getId();
		set.put("rawset", rawSetId);
		set.put("creationDate", DateUtils.now_ISO8601());
		set.put("set", actionSetId);
		set.put("enabled", "true");
		set.put("jobProperty", getJobProperty());

		token.getEnv().setAttribute(set.get("jobProperty"), set.get("rawset"));

		final String path = basePath + "/" + actionSetDirectory + "/" + rawSetId;
		log.info("using action set path: " + path);
		token.getEnv().setAttribute(getActionSetPathParam(), path);

		setList.add(set);
		final String sets = new Gson().toJson(setList);
		log.debug("built set: " + sets);

		token.getEnv().setAttribute("sets", sets);

		return Arc.DEFAULT_ARC;
	}

	/**
	 * Gets the action set id.
	 *
	 * @param token the token
	 * @return the action set id
	 */
	private String getActionSetId(final NodeToken token) {
		final String json = token.getEnv().getAttribute(getDedupConfigSequenceParam());
		final DedupConfigurationOrchestration dco = DedupConfigurationOrchestration.fromJSON(json);
		final String actionSetId = dco.getActionSetId();
		log.info("found actionSetId in workflow env: " + actionSetId);
		return actionSetId;
	}

	/**
	 * Gets the dedup config sequence param.
	 *
	 * @return the dedup config sequence param
	 */
	public String getDedupConfigSequenceParam() {
		return dedupConfigSequenceParam;
	}

	/**
	 * Sets the dedup config sequence param.
	 *
	 * @param dedupConfigSequenceParam the new dedup config sequence param
	 */
	public void setDedupConfigSequenceParam(final String dedupConfigSequenceParam) {
		this.dedupConfigSequenceParam = dedupConfigSequenceParam;
	}

	/**
	 * Gets the job property.
	 *
	 * @return the job property
	 */
	public String getJobProperty() {
		return jobProperty;
	}

	/**
	 * Sets the job property.
	 *
	 * @param jobProperty the new job property
	 */
	public void setJobProperty(final String jobProperty) {
		this.jobProperty = jobProperty;
	}

	public String getActionSetPathParam() {
		return actionSetPathParam;
	}

	public void setActionSetPathParam(final String actionSetPathParam) {
		this.actionSetPathParam = actionSetPathParam;
	}
}
