package eu.dnetlib.functionality.modular.ui.repositories;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import eu.dnetlib.data.collector.rmi.CollectorService;
import eu.dnetlib.data.collector.rmi.ProtocolDescriptor;
import eu.dnetlib.data.collector.rmi.ProtocolParameter;
import eu.dnetlib.enabling.datasources.DatasourceFunctions;
import eu.dnetlib.enabling.datasources.LocalOpenaireDatasourceManager;
import eu.dnetlib.enabling.datasources.common.Api;
import eu.dnetlib.enabling.datasources.common.ApiParam;
import eu.dnetlib.enabling.datasources.common.BrowsableField;
import eu.dnetlib.enabling.datasources.common.Datasource;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoInterfaceEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoMetaWfEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.SimpleParamEntry;
import eu.dnetlib.functionality.modular.ui.repositories.objects.VocabularyEntry;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants.WorkflowStatus;

public class RepoUIUtils {

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Autowired
	private LocalOpenaireDatasourceManager dsManager;

	@Value("${repo.ui.tickets.baseUrl}")
	private String ticketBaseUrl;

	private static final Log log = LogFactory.getLog(RepoUIUtils.class);

	private final Map<String, List<ProtocolParameter>> parametersMap = Maps.newHashMap();

	public RepoInterfaceEntry getApi(final String dsId, final String apiId) throws Exception {
		final RepoInterfaceEntry ifc = new RepoInterfaceEntry();

		final Datasource<?, ?, ?> ds = dsManager.getDs(dsId);

		final Api<?> api = dsManager.getApis(dsId)
			.stream()
			.filter(a -> a.getId().equals(apiId))
			.findFirst()
			.orElseThrow(() -> new Exception("Api not found"));

		final List<?> metaWfs = new ArrayList<>(); // TODO

		ifc.setId(api.getId());
		ifc.setLabel(String
			.format("%s (%s)", StringUtils.defaultIfBlank(ds.getEoscDatasourceType(), "-"), StringUtils.defaultIfBlank(api.getCompatibility(), "-")));
		ifc.setRemovable(api.getRemovable() && metaWfs.isEmpty());
		ifc.setRepoId(ds.getId());
		ifc.setRepoName(ds.getOfficialname());
		ifc.setRepoCountry(ds.getOrganizations()
			.stream()
			.map(o -> o.getCountry())
			.filter(o -> o != null)
			.findFirst()
			.orElse("UNKNOWN"));
		ifc.setRepoPrefix(ds.getNamespaceprefix());
		ifc.setRepoType(ds.getEoscDatasourceType());
		ifc.setEmail(ds.getContactemail());
		ifc.setActive(api.getActive());
		ifc.setProtocol(api.getProtocol());
		ifc.setConsentTermsOfUseLabel(BooleanUtils.toString(ds.getConsentTermsOfUse(), "YES", "NO", "UNKNOWN"));
		ifc.setFullTextDownloadLabel(BooleanUtils.toString(ds.getFullTextDownload(), "YES", "NO", "UNKNOWN"));
		ifc.setPrimaryProvideGateway(ds.getPrimaryProvideGateway());
		ifc.setAffiliatedProvideGateways(new LinkedHashSet<>(Arrays.asList(ds.getAffiliatedProvideGateways())));

		final Set<String> toVerifyParams = getParameterNamesForProtocol(ifc.getProtocol());
		ifc.getAccessParams().add(new SimpleParamEntry("baseUrl", api.getBaseurl()));

		for (final ApiParam p : api.getApiParams()) {
			if (toVerifyParams.contains(p.getParam())) {
				ifc.getAccessParams().add(new SimpleParamEntry(p.getParam(), p.getValue()));
				toVerifyParams.remove(p.getParam());
			} else {
				log.warn("Invalid param [" + p.getParam() + "] for protocol " + p.getValue() + " in ds " + dsId);
			}
		}

		for (final String pname : toVerifyParams) {
			ifc.getAccessParams().add(new SimpleParamEntry(pname, ""));
			log.info("Adding missing param [" + pname + "] for protocol " + api.getProtocol() + " in ds " + dsId);
		}

		ifc.setMetadataIdentifierPath(api.getMetadataIdentifierPath());
		ifc.setCompliance(api.getCompatibility());
		ifc.setComplianceOverrided(api.isCompatibilityOverrided());

		if (api.getLastAggregationDate() != null) {
			ifc.setAggrDate(DateUtils.calculate_ISO8601(api.getLastAggregationDate().getTime()));
			ifc.setAggrTotal(api.getLastAggregationTotal());
			ifc.setAggrMdId(api.getLastAggregationMdid());
			ifc.setAggrBackend(DatasourceFunctions.calculateMdstoreBackendById(api.getLastAggregationMdid()));
		}
		if (api.getLastCollectionDate() != null) {
			ifc.setCollDate(DateUtils.calculate_ISO8601(api.getLastCollectionDate().getTime()));
			ifc.setCollTotal(api.getLastCollectionTotal());
			ifc.setCollMdId(api.getLastCollectionMdid());
			ifc.setCollBackend(DatasourceFunctions.calculateMdstoreBackendById(api.getLastCollectionMdid()));
		}
		if (api.getLastDownloadDate() != null) {
			ifc.setDownloadDate(DateUtils.calculate_ISO8601(api.getLastDownloadDate().getTime()));
			ifc.setDownloadTotal(api.getLastDownloadTotal());
			ifc.setDownloadObjId(api.getLastDownloadObjid());
		}

		ifc.setTickets(findTicketsForDatasource(dsId));

		final StringTemplate st = new StringTemplate(IOUtils.toString(getClass()
			.getResourceAsStream("/eu/dnetlib/functionality/modular/ui/repositories/templates/getMetaWfs.xquery.st")));
		st.setAttribute("dsId", dsId);
		st.setAttribute("apiId", apiId);

		final String s = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(st.toString());

		final Document doc = new SAXReader().read(new StringReader(s));

		for (final Object o : doc.selectNodes("//metaWF")) {
			final Node n = (Node) o;

			final String id = n.valueOf("./id");
			final String name = n.valueOf("./name");
			final String status = n.valueOf("./status");
			final String repoByeWfId = n.valueOf("./destroyWorkflow");

			int progress = 0;
			try {
				switch (WorkflowStatus.valueOf(status)) {
				case MISSING:
					progress = 0;
					break;
				case ASSIGNED:
					progress = 25;
					break;
				case WAIT_SYS_SETTINGS:
					progress = 50;
					break;
				case WAIT_USER_SETTINGS:
					progress = 75;
					break;
				case EXECUTABLE:
					progress = 100;
					break;
				}
			} catch (final Exception e) {
				progress = 0;
			}
			ifc.getMetaWFs().add(new RepoMetaWfEntry(id, name, status, repoByeWfId, progress));
		}

		return ifc;
	}

	private List<BrowsableField> findTicketsForDatasource(final String dsId) throws Exception {
		return dsManager.listTickets(dsId)
			.stream()
			.sorted()
			.map(i -> new BrowsableField(ticketBaseUrl + i, "#" + i))
			.collect(Collectors.toList());
	}

	public List<BrowsableField> updateTicketsForDatasource(final String dsId, final Set<Integer> tickets) throws Exception {
		dsManager.updateTickets(dsId, tickets);

		return findTicketsForDatasource(dsId);
	}

	public List<VocabularyEntry> fetchVocabularyTerms(final String voc) throws ISLookUpException {
		final String xquery = "for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType')[.//VOCABULARY_NAME/@code = '"
			+ voc.trim() + "']//TERM return concat($x/@code, ' @@@ ', $x/@english_name)";

		final List<VocabularyEntry> list = Lists.newArrayList();
		for (final String s : serviceLocator.getService(ISLookUpService.class).quickSearchProfile(xquery)) {
			final String[] arr = s.split("@@@");
			list.add(new VocabularyEntry(arr[0].trim(), arr[1].trim()));
		}
		Collections.sort(list);

		return list;
	}

	private Set<String> getParameterNamesForProtocol(final String protocol) {
		if (parametersMap.isEmpty()) {
			for (final ProtocolDescriptor d : serviceLocator.getService(CollectorService.class).listProtocols()) {
				parametersMap.put(d.getName().toLowerCase(), d.getParams());
			}
		}
		final Set<String> res = Sets.newHashSet();
		if (parametersMap.containsKey(protocol.toLowerCase())) {
			for (final ProtocolParameter p : parametersMap.get(protocol.toLowerCase())) {
				res.add(p.getName());
			}
		}

		return res;
	}

}
