package eu.dnetlib.functionality.modular.ui.repositories;

import java.io.IOException;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.annotate.JsonSerialize.Inclusion;
import org.codehaus.jackson.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.client.RestTemplate;

import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.data.collector.rmi.CollectorService;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.ProtocolParameterValue;
import eu.dnetlib.enabling.datasources.LocalOpenaireDatasourceManager;
import eu.dnetlib.enabling.datasources.OpenaireDatasource;
import eu.dnetlib.enabling.datasources.common.Api;
import eu.dnetlib.enabling.datasources.common.ApiParam;
import eu.dnetlib.enabling.datasources.common.ApiParamImpl;
import eu.dnetlib.enabling.datasources.common.BrowsableField;
import eu.dnetlib.enabling.datasources.common.BrowseTerm;
import eu.dnetlib.enabling.datasources.common.Datasource;
import eu.dnetlib.enabling.datasources.common.DsmException;
import eu.dnetlib.enabling.datasources.common.Identity;
import eu.dnetlib.enabling.datasources.common.Organization;
import eu.dnetlib.enabling.datasources.common.PidSystem;
import eu.dnetlib.enabling.datasources.common.SearchApisEntry;
import eu.dnetlib.enabling.datasources.common.SimpleDatasource;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.AbstractAjaxController;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoInterfaceEntry;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.WorkflowSectionGrouper;
import eu.dnetlib.msro.workflows.sarasvati.loader.WorkflowExecutor;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

@Controller
public class RepoInternalController extends AbstractAjaxController {

	@Autowired
	private LocalOpenaireDatasourceManager dsManager;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private WorkflowSectionGrouper workflowSectionGrouper;

	@Resource
	private WorkflowExecutor workflowExecutor;

	@Resource
	private RepoUIUtils repoUIUtils;

	@Value("${repo.ui.mdstore.sync.button}")
	private String syncButton;

	@Value("${repo.ui.mdstore.sync.remote.desc}")
	private String syncDesc;

	@Value("${repo.ui.mdstore.sync.remote.api.baseUrl}")
	private String syncApiBaseUrl;

	@Value("${repo.ui.mdstore.sync.remote.is.baseUrl}")
	private String syncISBaseUrl;

	@Value("${repo.ui.mdstore.sync.local.mongodb.url}")
	private String mongoUrl;

	@Value("${services.mdstore.mongodb.db}")
	private String mongoDb;

	private static final Log log = LogFactory.getLog(RepoInternalController.class);

	@RequestMapping(value = "/ui/browseRepoField.do")
	public @ResponseBody List<? extends BrowseTerm> browseRepoField(@RequestParam(value = "field", required = true) final String field) throws Exception {
		return dsManager.browseField(field);
	}

	@Cacheable(cacheNames = "repoUIJsonCache", key = "#param, #value", condition = "#refresh == false")
	@RequestMapping(value = "/ui/listApis.do")
	public @ResponseBody List<? extends SearchApisEntry> listApis(
		@RequestParam(value = "param", required = true) final String param,
		@RequestParam(value = "value", required = true) final String value,
		@RequestParam(value = "refresh", required = false) final String refresh) throws Exception {

		return dsManager.searchApis(param, value);
	}

	@RequestMapping(value = "/ui/listRepositories.json")
	public @ResponseBody List<SimpleDatasource> listRepositories(@RequestParam(value = "type", required = true) final String type) throws Exception {
		return dsManager.searchDatasourcesByType(type);
	}

	@CacheEvict("repoUIJsonCache")
	@RequestMapping(value = "/ui/validateRepo.do")
	public @ResponseBody String listRepositories(@RequestParam(value = "id", required = true) final String id,
		@RequestParam(value = "b", required = true) final boolean b) throws Exception {

		final String query = "count(/*[.//RESOURCE_TYPE/@value='MetaWorkflowDSResourceType' and .//DATAPROVIDER/@id='" + id + "'])";
		if (!b && Integer.parseInt(serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query)) > 0) {
			throw new Exception("Repo " + id
				+ " can be invalidated: it is related to some metawfs");
		}

		final String newId = b ? serviceLocator.getService(ISRegistryService.class).validateProfile(id)
			: serviceLocator.getService(ISRegistryService.class).invalidateProfile(id);

		return newId;
	}

	@RequestMapping(value = "/ui/getRepoDetails.do")
	public void getRepoDetails(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id)
		throws Exception {
		final Datasource<?, ?, ?> ds = dsManager.getDs(id);
		final List<? extends Api<?>> apis = dsManager.getApis(id);

		final ObjectMapper mapper = new ObjectMapper();
		mapper.setSerializationInclusion(Inclusion.NON_EMPTY);

		final JsonNode jsonDs = mapper.valueToTree(ds);
		((ObjectNode) jsonDs).put("apis", mapper.valueToTree(apis));

		response.setContentType("application/json;charset=UTF-8");

		IOUtils.write(jsonDs.toString(), response.getOutputStream());
	}

	@RequestMapping("/ui/repoMetaWf.new")
	public @ResponseBody String newDataProviderWorkflow(@RequestParam(value = "id", required = true) final String repoId,
		@RequestParam(value = "name", required = true) final String repoName,
		@RequestParam(value = "iface", required = true) final String ifaceId,
		@RequestParam(value = "wf", required = true) final String wfId) throws Exception {

		final Map<String, Object> params = Maps.newHashMap();
		params.put(WorkflowsConstants.DATAPROVIDER_ID, repoId);
		params.put(WorkflowsConstants.DATAPROVIDER_NAME, repoName);
		params.put(WorkflowsConstants.DATAPROVIDER_INTERFACE, ifaceId);

		return workflowExecutor.startProcess(wfId, params);
	}

	@RequestMapping("/ui/repoMetaWf.destroy")
	public @ResponseBody String destroyDataProviderWorkflow(@RequestParam(value = "destroyWf", required = true) final String destroyWfId)
		throws Exception {
		return workflowExecutor.startProcess(destroyWfId, null);
	}

	@RequestMapping("/ui/repoApi.get")
	public @ResponseBody RepoInterfaceEntry getRepoApi(@RequestParam(value = "repoId", required = true) final String repoId,
		@RequestParam(value = "ifaceId", required = true) final String ifaceId) throws Exception {

		return repoUIUtils.getApi(repoId, ifaceId);
	}

	@RequestMapping("/ui/repoApi.update")
	public @ResponseBody boolean updateRepoApi(
		@RequestParam(value = "id", required = true) final String repoId,
		@RequestParam(value = "iface", required = true) final String ifaceId,
		@RequestParam(value = "accessParams", required = false) final String accessParamsJson,
		@RequestParam(value = "mdIdPath", required = false) final String mdIdPath) throws Exception {

		if (!StringUtils.isEmpty(accessParamsJson)) {
			final Map<String, String> params = new Gson().fromJson(accessParamsJson, new TypeToken<Map<String, String>>() {}.getType());
			final String baseUrl = params.remove("baseUrl");
			dsManager.updateApiDetails(repoId, ifaceId, mdIdPath, baseUrl, params);
		}

		return true;
	}

	@RequestMapping("/ui/repoApi.delete")
	public @ResponseBody boolean deleteRepoApi(
		@RequestParam(value = "repo", required = true) final String repoId,
		@RequestParam(value = "iface", required = true) final String ifaceId) throws Exception {
		dsManager.deleteApi(repoId, ifaceId);
		return true;
	}

	@CacheEvict("repoUIJsonCache")
	@RequestMapping("/ui/repoApiCompliance.update")
	public @ResponseBody boolean updateRepoApiCompliance(@RequestParam(value = "id", required = true) final String repoId,
		@RequestParam(value = "iface", required = true) final String ifaceId,
		@RequestParam(value = "compliance", required = true) final String compliance) throws Exception {

		log.debug("SET COMPLIANCE TO " + compliance);

		dsManager.updateCompliance(repoId, ifaceId, compliance, true);

		return true;
	}

	@CacheEvict("repoUIJsonCache")
	@RequestMapping("/ui/repoApiCompliance.reset")
	public @ResponseBody boolean resetRepoApiCompliance(@RequestParam(value = "id", required = true) final String repoId,
		@RequestParam(value = "iface", required = true) final String ifaceId) throws Exception {

		log.debug("RESET COMPLIANCE");

		dsManager.updateCompliance(repoId, ifaceId, null, true);

		return true;
	}

	@RequestMapping("/ui/repos/repoApi.html")
	public void repoApiHtml(final ModelMap map) throws Exception {
		if (StringUtils.isNoneBlank(syncButton, syncDesc, syncApiBaseUrl, syncISBaseUrl)) {
			map.addAttribute("syncButton", syncButton);
			map.addAttribute("syncDesc", syncDesc);
			map.addAttribute("syncISBaseUrl", syncISBaseUrl);
			map.addAttribute("mongoDB", mongoDb);
			map.addAttribute("mongoUrl", mongoUrl);
		}
	}

	@RequestMapping("/ui/repoApi.new")
	public @ResponseBody boolean addRepoApi(@RequestParam(value = "repoId", required = true) final String repoId,
		@RequestParam(value = "iface", required = true) final String ifaceJson) throws DsmException {
		final Api<ApiParam> iface = new Gson().fromJson(ifaceJson, new TypeToken<Api<ApiParamImpl>>() {}.getType());

		iface.setDatasource(repoId);

		log.info("Adding api " + iface.getId() + " to repository " + repoId);

		dsManager.addApi(iface);

		return true;
	}

	@RequestMapping("/ui/repo.new")
	public @ResponseBody boolean addRepo(@RequestParam(value = "repo", required = true) final String repoJson) throws DsmException {
		final Datasource<Organization<?>, Identity, PidSystem> ds = new Gson().fromJson(repoJson, OpenaireDatasource.class);
		final Date now = new Date();
		ds.setDateofcollection(new java.sql.Date(now.getTime()));

		if (StringUtils.isBlank(ds.getPrimaryProvideGateway())) {
			ds.setPrimaryProvideGateway("openaire");
		}

		if (ds.getAffiliatedProvideGateways() == null) {
			ds.setAffiliatedProvideGateways(new String[] {});
		}

		if (StringUtils.isBlank(ds.getEnglishname())) {
			ds.setEnglishname(ds.getOfficialname());
		}

		if (StringUtils.isBlank(ds.getProvenanceaction())) {
			ds.setProvenanceaction("user:insert");
		}

		log.info("Adding datasource " + ds.getId() + " - name " + ds.getOfficialname());

		dsManager.saveDs(ds);

		return true;
	}

	@RequestMapping("/ui/listValidValuesForParam.do")
	public @ResponseBody List<ProtocolParameterValue> listValidValuesForParam(
		@RequestParam(value = "protocol", required = true) final String protocol,
		@RequestParam(value = "param", required = true) final String param,
		@RequestParam(value = "baseUrl", required = true) final String baseUrl) throws CollectorServiceException {

		return serviceLocator.getService(CollectorService.class).listValidValuesForParam(protocol, baseUrl, param, null);
	}

	@RequestMapping(value = "/ui/remoteDatasource/apis", method = RequestMethod.GET)
	public void listRemoteApis(@RequestParam(value = "repo", required = true) final String repoId,
		final HttpServletResponse response) throws IOException {

		final RestTemplate restTemplate = new RestTemplate();

		final String res = restTemplate.getForObject(syncApiBaseUrl + "/ds/api/" + repoId, String.class);

		response.setContentType("application/json;charset=UTF-8");

		IOUtils.write(res, response.getOutputStream());
	}

	@RequestMapping(value = "/ui/remoteDatasource/api", method = RequestMethod.POST)
	public void saveRemoteApi(@RequestBody final Api<ApiParamImpl> api, final HttpServletResponse response) throws IOException {

		final String url = syncApiBaseUrl + "/ds/api/add";

		log.info("Adding new remote API:");
		log.info(" - repo id: " + api.getDatasource());
		log.info(" - api id: " + api.getId());
		log.info(" - rest call: " + url);
		log.info(" - json: " + new Gson().toJson(api));

		final RestTemplate restTemplate = new RestTemplate();

		restTemplate.postForObject(url, api, Void.class);

		listRemoteApis(api.getDatasource(), response);
	}

	@ResponseBody
	@RequestMapping(value = "/ui/updateRepoTickets.do", method = RequestMethod.POST)
	public List<BrowsableField> updateRepoTickets(@RequestParam(name = "id", required = true) final String repoId, @RequestBody final List<Integer> tickets)
		throws Exception {
		return repoUIUtils.updateTicketsForDatasource(repoId, new LinkedHashSet<>(tickets));
	}

}
