package eu.dnetlib.msro.workflows.blacklist;

import java.io.StringReader;
import java.util.List;
import java.util.Map;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.data.hadoop.rmi.HadoopService;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.hadoop.rmi.hbase.Column;
import eu.dnetlib.data.hadoop.rmi.hbase.HBaseRowDescriptor;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.transform.xml.AbstractDNetXsltFunctions;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.openaire.blacklist.BlacklistManager;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Created by alessia on 23/10/15.
 */
public class ApplyBlacklistJobNode extends SimpleJobNode {

	private static final Log log = LogFactory.getLog(ApplyBlacklistJobNode.class);

	@Autowired
    private BlacklistManager blacklistManager;

    @Autowired
    private UniqueServiceLocator serviceLocator;

    @Autowired
    private ResultSetClientFactory resultSetClientFactory;

	private String clusterName;

    private String tableName;

    @Override
    protected String execute(NodeToken token) throws Exception {
        final W3CEndpointReference epr = this.blacklistManager.getAcceptedBlacklistEntries();
	    final HadoopService hadoopService = serviceLocator.getService(HadoopService.class);

	    final SAXReader r = new SAXReader();

	    final Map<String, String> relCache = Maps.newHashMap();
	    int count = 0;

	    for(String s : resultSetClientFactory.getClient(epr)) {
		    final Document doc = r.read(new StringReader(s));

		    if (log.isDebugEnabled()) {
			    log.debug(doc.asXML());
		    }

		    final ResolvedBlacklistEntry rbeRoot = new ResolvedBlacklistEntry();

		    final String source = fullId(doc.valueOf("//FIELD[./@name='source_type']"), doc.valueOf("//FIELD[./@name='source']"));
		    rbeRoot.setSource(getResolvedId(source, hadoopService.describeHBaseColumn(getClusterName(), getTableName(), source).getColumns()));

		    final String target = fullId(doc.valueOf("//FIELD[./@name='target_type']"), doc.valueOf("//FIELD[./@name='target']"));
		    rbeRoot.setTarget(getResolvedId(target, hadoopService.describeHBaseColumn(getClusterName(), getTableName(), target).getColumns()));

		    final String relType = doc.valueOf("//FIELD[./@name='relationship']");
		    rbeRoot.setRelType(relType);

		    final ResolvedBlacklistEntry rbeNative = new ResolvedBlacklistEntry(source, target, relType);

		    applyBlackListEntry(hadoopService, relCache, rbeRoot);
		    applyBlackListEntry(hadoopService, relCache, rbeNative);

		      count++;
	    }

	    log.info("blacklist size: " + count);
	    token.getEnv().setAttribute("blacklistSize", count);

        return Arc.DEFAULT_ARC;
    }

	private void applyBlackListEntry(final HadoopService hadoopService, final Map<String, String> relCache, final ResolvedBlacklistEntry rbe)
			throws ISLookUpException, HadoopServiceException {
		if (!relCache.containsKey(rbe.getRelType())) {
			relCache.put(rbe.getRelType(), fetchInverse(rbe.getRelType()));
		}

		hadoopService.deleteHBaseColumn(getClusterName(), getTableName(), getHBaseRowDescriptor(rbe.getSource(), rbe.getTarget(), rbe.getRelType()));

		hadoopService.deleteHBaseColumn(getClusterName(), getTableName(), getHBaseRowDescriptor(rbe.getTarget(), rbe.getSource(), relCache.get(
				rbe.getRelType())));
	}

	private HBaseRowDescriptor getHBaseRowDescriptor(final String source, final String target, final String relType) {

		final List<Column> cols = Lists.newArrayList(new Column(relType, Lists.newArrayList(target)));
		return new HBaseRowDescriptor(source, cols);
	}

	private String fetchInverse(final String relType) throws ISLookUpException {
		final String xquery = "let $x:= /RESOURCE_PROFILE["
				+ " .//RESOURCE_TYPE/@value = 'OntologyDSResourceType' and "
				+ " .//TERM/@encoding='"+relType+"']"
				+ "let $y:= $x//TERM[./@encoding='"+relType+"']//RELATION[./@type='inverseOf']/@code/string() "
				+ "return $x//TERM[./@code = $y]/@encoding/string()";
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
	}

	private String fullId(final String type, final String id) {
		final String fullId = AbstractDNetXsltFunctions.oafSimpleId(type, id);

		return OafRowKeyDecoder.decode(fullId).getKey();
	}

	private String getResolvedId(String originalId, List<Column> columns) {
		if (StringUtils.isBlank(originalId)) {
			throw new IllegalArgumentException("cannot resolve an empty id");
		}

	    for (Column c : columns) {
            if (c.getFamily().contains("isMergedIn")) {

	            for (String s : c.getQualifier()) {
		            if (StringUtils.isNotBlank(s)) return s;
	            }
            }
        }
        return originalId;
    }

	public String getClusterName() {
		return clusterName;
	}

	public void setClusterName(final String clusterName) {
		this.clusterName = clusterName;
	}

	public String getTableName() {
		return tableName;
	}

	public void setTableName(final String tableName) {
		this.tableName = tableName;
	}

}
