package eu.dnetlib.openaire.blacklist;

import java.io.StringReader;
import java.util.List;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;

import eu.dnetlib.functionality.index.client.solr.SolrIndexClient;
import eu.dnetlib.functionality.index.client.solr.SolrIndexClientFactory;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * The goal of this task is to return the original identifiers of objects merged in a representative object by deduplication. Created by
 * alessia on 09/02/16.
 */
public class OpenaireIdResolver {

	private static final Log log = LogFactory.getLog(OpenaireIdResolver.class);
	public final static String SOLR_COLLECTION_POSTFIX = "-index-openaire";
	private final static String RESULT_FIELD = "__result";
	private final static String XPATH_TO_MERGED = "//*[local-name()='entity']/*//children/result/@objidentifier";
	private final SAXReader saxReader = new SAXReader();
	@Autowired
	private UniqueServiceLocator serviceLocator;
	@Autowired
	private SolrIndexClientFactory indexClientFactory;

	public Set<String> resolveIdentifier(final String id) {
		if (StringUtils.isBlank(id)) return  Sets.newHashSet();
		else {
			return findOriginalIds(id);
		}
	}

	protected Set<String> findOriginalIds(final String id) {
		try(final SolrIndexClient client = (SolrIndexClient) indexClientFactory.getClient(getPublicIndexCollection()+SOLR_COLLECTION_POSTFIX)) {

			final String query = String.format("objidentifier:\"%s\"", id);
			final QueryResponse response = client.query(query, 1);
			final SolrDocumentList results = response.getResults();
			if (results.isEmpty()) {
				log.debug("Query " + query + " returned 0 documents");
				return Sets.newHashSet();
			}
			// my results contain the document with the given identifier
			final SolrDocument solrDoc = results.get(0);
			return extractMergedIdentifiers(solrDoc);
		} catch (final Exception e) {
			log.error("Can't get original ids for " + id + "\n ", e);
			throw new RuntimeException("Can't get original ids for " + id + "\n " + e);
		}
	}

	@SuppressWarnings("unchecked")
	protected Set<String> extractMergedIdentifiers(final SolrDocument doc) throws DocumentException {
		final String xmlRecord = (String) doc.getFirstValue(RESULT_FIELD);
		final Document xmlDoc = this.saxReader.read(new StringReader(xmlRecord));
		return Sets.newHashSet(Iterables.transform(xmlDoc.selectNodes(XPATH_TO_MERGED), (Function<Attribute, String>) a -> a.getStringValue()));
	}

	protected String getIndexEndpoint() throws ISLookUpException {
		return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"for $x in collection('/db/DRIVER/ServiceResources/IndexServiceResourceType') return $x//PROTOCOL[./@name='solr']/@address/string()");
	}

	protected String getPublicIndexCollection() throws ISLookUpException {
		return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"for $x in collection('/db/DRIVER/ServiceResources/SearchServiceResourceType') return $x[.//PROPERTY[@key='infrastructure']/@value='public']//PROPERTY[@key='mdformat']/@value/string()");
	}

	public UniqueServiceLocator getServiceLocator() {
		return serviceLocator;
	}

	public void setServiceLocator(final UniqueServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}

}
