package eu.dnetlib.enabling.datasources;

import java.sql.Array;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.enabling.datasources.common.Api;
import eu.dnetlib.enabling.datasources.common.ApiParam;
import eu.dnetlib.enabling.datasources.common.ApiParamImpl;
import eu.dnetlib.enabling.datasources.common.Datasource;
import eu.dnetlib.enabling.datasources.common.Identity;
import eu.dnetlib.enabling.datasources.common.Organization;
import eu.dnetlib.enabling.datasources.common.PidSystem;
import eu.dnetlib.enabling.datasources.common.SearchApisEntry;
import eu.dnetlib.enabling.datasources.common.SimpleDatasource;
import eu.dnetlib.miscutils.datetime.DateUtils;

public class DatasourceFunctions {

	private static final String EOSC_DATA_SOURCE = "Data Source";

	private static final Log log = LogFactory.getLog(DatasourceFunctions.class);

	public static SimpleDatasource mapToSimpleDs(final Map<String, Object> map) {
		final SimpleDatasource ds = new SimpleDatasource();
		ds.setId(castObject(map.get("id"), String.class));
		ds.setName(castObject(map.get("name"), String.class));
		ds.setOrigId(castObject(map.get("id"), String.class));
		ds.setEoscDatasourceType(castObject(map.get("eosc_datasource_type"), String.class));
		ds.setValid(true);
		ds.setPrimaryProvideGateway(castObject(map.get("primary_provide_gateway"), String.class));
		ds.setAffiliatedProvideGateways(Arrays.stream(castObject(map.get("affiliated_provide_gateways"), String.class).split(","))
			.filter(StringUtils::isNotBlank)
			.collect(Collectors.toSet()));

		try {
			final Array arr = castObject(map.get("apis"), Array.class);

			if (arr != null && ((Object[]) arr.getArray()).length > 0) {
				ds.setApis(Arrays.stream((Object[]) arr.getArray())
					.filter(Objects::nonNull)
					.map(Object::toString)
					.collect(Collectors.toSet()));
			}
		} catch (final SQLException e) {
			log.error("Error parsing array (apis)", e);
			throw new RuntimeException("Error parsing array (apis)", e);
		}

		return ds;
	}

	public static Datasource<Organization<?>, Identity, PidSystem> mapToDatasource(final Map<String, Object> map) {
		final Datasource<Organization<?>, Identity, PidSystem> ds = new OpenaireDatasource();
		ds.setId(castObject(map.get("id"), String.class));
		ds.setOfficialname(castObject(map.get("officialname"), String.class));
		ds.setEnglishname(castObject(map.get("englishname"), String.class));
		ds.setWebsiteurl(castObject(map.get("websiteurl"), String.class));
		ds.setLogourl(castObject(map.get("logourl"), String.class));
		ds.setContactemail(castObject(map.get("contactemail"), String.class));
		ds.setLatitude(castObject(map.get("latitude"), Double.class));
		ds.setLongitude(castObject(map.get("longitude"), Double.class));
		ds.setTimezone(castObject(map.get("timezone"), String.class));
		ds.setNamespaceprefix(castObject(map.get("namespaceprefix"), String.class));
		ds.setLanguages(castObject(map.get("languages"), String.class));
		ds.setCollectedfrom(castObject(map.get("collectedfrom"), String.class));
		ds.setDateofvalidation(castObject(map.get("dateofvalidation"), Date.class));
		ds.setEoscDatasourceType(castObject(map.get("eosc_datasource_type"), String.class));
		ds.setProvenanceaction(castObject(map.get("provenanceaction"), String.class));
		ds.setDateofcollection(castObject(map.get("dateofcollection"), Date.class));
		ds.setPlatform(castObject(map.get("platform"), String.class));
		ds.setActivationId(castObject(map.get("activationId"), String.class));
		ds.setDescription(castObject(map.get("description"), String.class));
		ds.setReleasestartdate(castObject(map.get("releasestartdate"), Date.class));
		ds.setReleaseenddate(castObject(map.get("releaseenddate"), Date.class));
		ds.setMissionstatementurl(castObject(map.get("missionstatementurl"), String.class));
		ds.setDatabaseaccesstype(castObject(map.get("databaseaccesstype"), String.class));
		ds.setDatauploadtype(castObject(map.get("datauploadtype"), String.class));
		ds.setDatabaseaccessrestriction(castObject(map.get("databaseaccessrestriction"), String.class));
		ds.setDatauploadrestriction(castObject(map.get("datauploadrestriction"), String.class));
		ds.setCitationguidelineurl(castObject(map.get("citationguidelineurl"), String.class));
		ds.setCertificates(castObject(map.get("certificates"), String.class));
		ds.setAggregator(castObject(map.get("aggregator"), String.class));
		ds.setIssn(castObject(map.get("issn"), String.class));
		ds.setEissn(castObject(map.get("eissn"), String.class));
		ds.setLissn(castObject(map.get("lissn"), String.class));
		ds.setRegisteredby(castObject(map.get("registeredby"), String.class));
		ds.setSubjects(castObject(map.get("subjects"), String.class));
		ds.setManaged(castObject(map.get("managed"), Boolean.class));
		ds.setConsentTermsOfUse(castObject(map.get("consenttermsofuse"), Boolean.class));
		ds.setConsentTermsOfUseDate(castObject(map.get("consenttermsofusedate"), Date.class));
		ds.setFullTextDownload(castObject(map.get("fulltextdownload"), Boolean.class));
		ds.setStatus(castObject(map.get("status"), String.class));
		ds.setPrimaryProvideGateway(castObject(map.get("primary_provide_gateway"), String.class));
		ds.setAffiliatedProvideGateways(castObject(map.get("affiliated_provide_gateways"), String.class).split(","));

		return ds;
	}

	public static Identity mapToDsIdentity(final Map<String, Object> map) {
		final Identity id = new Identity();
		id.setPid(castObject(map.get("pid"), String.class));
		id.setIssuertype(castObject(map.get("issuertype"), String.class));
		return id;
	}

	public static Organization<Datasource<?, ?, ?>> mapToDsOrganization(final Map<String, Object> map) {
		final Organization<Datasource<?, ?, ?>> org = new Organization<>();
		org.setId(castObject(map.get("id"), String.class));
		org.setLegalshortname(castObject(map.get("legalshortname"), String.class));
		org.setLegalname(castObject(map.get("legalname"), String.class));
		org.setWebsiteurl(castObject(map.get("websiteurl"), String.class));
		org.setLogourl(castObject(map.get("logourl"), String.class));
		org.setCountry(castObject(map.get("country"), String.class));
		org.setCollectedfrom(castObject(map.get("collectedfrom"), String.class));
		org.setDateofcollection(castObject(map.get("dateofcollection"), Date.class));
		org.setProvenanceaction(castObject(map.get("provenanceaction"), String.class));
		return org;
	}

	public static SearchApisEntry mapToSearchApisEntry(final Map<String, Object> map) {
		final SearchApisEntry a = new SearchApisEntry();
		a.setId(castObject(map.get("id"), String.class));
		a.setCompliance(castObject(map.get("compliance"), String.class));
		a.setActive(castObject(map.get("active"), Boolean.class));
		a.setRepoId(castObject(map.get("dsId"), String.class));
		a.setRepoName(castObject(map.get("name"), String.class));
		a.setRepoCountry(castObject(map.get("country"), String.class));
		a.setRepoPrefix(castObject(map.get("prefix"), String.class));

		if (map.get("aggrDate") != null) {
			final Timestamp t = castObject(map.get("aggrDate"), Timestamp.class);
			a.setAggrDate(DateUtils.calculate_ISO8601(t.getTime()));
		} else {
			a.setAggrDate("");
		}

		a.setAggrTotal(castObject(map.get("aggrTotal"), Integer.class));
		a.setAggrBackend(calculateMdstoreBackendById(castObject(map.get("aggrMdId"), String.class)));

		if (map.get("collDate") != null) {
			final Timestamp t = castObject(map.get("collDate"), Timestamp.class);
			a.setCollDate(DateUtils.calculate_ISO8601(t.getTime()));
		} else {
			a.setCollDate("");
		}

		a.setCollTotal(castObject(map.get("collTotal"), Integer.class));
		a.setCollBackend(calculateMdstoreBackendById(castObject(map.get("collMdId"), String.class)));
		a.setProtocol(castObject(map.get("protocol"), String.class));
		a.setAlternativeName(castObject(map.get("alternativeName"), String.class));
		a.setRepoOrganization(castObject(map.get("organization"), String.class));
		a.setPrimaryProvideGateway(castObject(map.get("primary_provide_gateway"), String.class));
		a.setAffiliatedProvideGateways(Arrays.stream(castObject(map.get("affiliated_provide_gateways"), String.class).split(","))
			.filter(StringUtils::isNotBlank)
			.collect(Collectors.toSet()));

		return a;
	}

	public static String calculateMdstoreBackendById(final String mdId) {
		if (StringUtils.isBlank(mdId)) {
			return "";
		} else if (StringUtils.trim(mdId).startsWith("md-") && StringUtils.trim(mdId).length() <= 40) {
			return MDStoreBackend.HDFS.toString();
		} else {
			return MDStoreBackend.MONGO.toString();
		}
	}

	public static Api<ApiParam> mapToApi(final Map<String, Object> map) {
		final Api<ApiParam> a = new Api<>();
		a.setId(castObject(map.get("id"), String.class));
		a.setProtocol(castObject(map.get("protocol"), String.class));
		a.setDatasource(castObject(map.get("datasource"), String.class));
		a.setContentdescription(castObject(map.get("contentdescription"), String.class));
		a.setActive(castObject(map.get("active"), Boolean.class));
		a.setRemovable(castObject(map.get("removable"), Boolean.class));
		a.setCompatibility(castObject(map.get("compatibility"), String.class));
		a.setCompatibilityOverrided(castObject(map.get("isCompatibilityOverrided"), Boolean.class));
		a.setMetadataIdentifierPath(castObject(map.get("metadataIdentifierPath"), String.class));
		a.setLastCollectionTotal(castObject(map.get("lastCollectionTotal"), Integer.class));
		a.setLastCollectionDate(castObject(map.get("lastCollectionDate"), Timestamp.class));
		a.setLastCollectionMdid(castObject(map.get("lastCollectionMdid"), String.class));
		a.setLastAggregationTotal(castObject(map.get("lastAggregationTotal"), Integer.class));
		a.setLastAggregationDate(castObject(map.get("lastAggregationDate"), Timestamp.class));
		a.setLastAggregationMdid(castObject(map.get("lastAggregationMdid"), String.class));
		a.setLastDownloadTotal(castObject(map.get("lastDownloadTotal"), Integer.class));
		a.setLastDownloadDate(castObject(map.get("lastDownloadDate"), Timestamp.class));
		a.setLastDownloadObjid(castObject(map.get("lastDownloadObjid"), String.class));
		a.setLastValidationJob(castObject(map.get("lastValidationJob"), String.class));
		a.setBaseurl(castObject(map.get("baseUrl"), String.class));
		try {
			final Array arr = castObject(map.get("params"), Array.class);
			if (arr != null) {
				a.setApiParams(Arrays.stream((Object[]) arr.getArray())
					.filter(Objects::nonNull)
					.map(Object::toString)
					.map(s -> {
						final ApiParam p = new ApiParamImpl();
						p.setParam(StringUtils.substringBefore(s, "="));
						p.setValue(StringUtils.substringAfter(s, "="));
						return p;
					})
					.collect(Collectors.toSet()));
			}
		} catch (final SQLException e) {
			log.error("Error parsing array params", e);
			throw new RuntimeException("Error parsing array params", e);
		}

		return a;
	}

	public static Map<String, Object> dsToMap(final Datasource<Organization<?>, Identity, PidSystem> ds) {
		final Map<String, Object> map = new HashMap<>();

		map.put("id", ds.getId());
		map.put("officialname", ds.getOfficialname());
		map.put("englishname", ds.getEnglishname());
		map.put("websiteurl", ds.getWebsiteurl());
		map.put("logourl", ds.getLogourl());
		map.put("contactemail", ds.getContactemail());
		map.put("latitude", ds.getLatitude());
		map.put("longitude", ds.getLongitude());
		map.put("timezone", ds.getTimezone());
		map.put("namespaceprefix", ds.getNamespaceprefix());
		map.put("languages", ds.getLanguages());
		map.put("collectedfrom", ds.getCollectedfrom());
		map.put("dateofvalidation", ds.getDateofvalidation());
		map.put("eosctype", EOSC_DATA_SOURCE);
		map.put("eoscdatasourcetype", ds.getEoscDatasourceType());
		map.put("provenanceaction", ds.getProvenanceaction());
		map.put("platform", ds.getPlatform());
		map.put("activationid", ds.getActivationId());
		map.put("description", ds.getDescription());
		map.put("releasestartdate", ds.getReleasestartdate());
		map.put("releaseenddate", ds.getReleaseenddate());
		map.put("missionstatementurl", ds.getMissionstatementurl());
		map.put("databaseaccesstype", ds.getDatabaseaccesstype());
		map.put("datauploadtype", ds.getDatauploadtype());
		map.put("databaseaccessrestriction", ds.getDatabaseaccessrestriction());
		map.put("datauploadrestriction", ds.getDatauploadrestriction());
		map.put("citationguidelineurl", ds.getCitationguidelineurl());
		map.put("pidsystems", ds.getPidsystems());
		map.put("certificates", ds.getCertificates());
		map.put("aggregator", ds.getAggregator());
		map.put("issn", ds.getIssn());
		map.put("eissn", ds.getEissn());
		map.put("lissn", ds.getLissn());
		map.put("registeredby", ds.getRegisteredby());
		map.put("subjects", ds.getSubjects());
		map.put("managed", ds.getManaged());
		map.put("status", ds.getStatus());
		map.put("primaryprovidegateway", ds.getPrimaryProvideGateway());
		map.put("affiliatedprovidegateways", StringUtils.join(ds.getAffiliatedProvideGateways(), ","));

		return map;
	}

	public static Map<String, Object> orgToMap(final String dsId, final Organization<?> org) {
		final Map<String, Object> map = new HashMap<>();
		map.put("dsId", dsId);
		map.put("orgId", org.getId());
		map.put("legalname", org.getLegalname());
		map.put("legalshortname", org.getLegalshortname());
		map.put("websiteurl", org.getWebsiteurl());
		map.put("logourl", org.getLogourl());
		map.put("country", org.getCountry());
		map.put("collectedfrom", org.getCollectedfrom());
		return map;
	}

	public static Map<String, Object> apiToMap(final Api<?> api) {
		final Map<String, Object> map = new HashMap<>();
		map.put("apiId", api.getId());
		map.put("protocol", api.getProtocol());
		map.put("baseUrl", api.getBaseurl());
		map.put("dsId", api.getDatasource());
		map.put("contentDescription", api.getContentdescription());
		map.put("compatibility", api.getCompatibility());
		map.put("metadataIdentifierPath", api.getMetadataIdentifierPath());
		map.put("removable", api.getRemovable());
		// The other fields are not required in the INSERT operation
		return map;
	}

	@SuppressWarnings("unchecked")
	private static <T> T castObject(final Object o, final Class<T> clazz) {
		if (o == null) { return null; }
		if (clazz.isInstance(o)) { return (T) o; }
		throw new IllegalArgumentException("Type not managed: " + clazz.getSimpleName());
	}

}
