package eu.dnetlib.msro.openaireplus.workflows.nodes.index;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Resource;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.data.index.CloudIndexClientException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.AsyncJobNode;
import eu.dnetlib.openaire.directindex.api.RecentResultsQueue;
import eu.dnetlib.openaire.directindex.utils.OafToIndexRecordFactory;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.SolrInputDocument;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;

/**
 * Created by michele on 15/12/15.
 */
public class FeedMissingClaimsJobNode extends AsyncJobNode {

    private static final Log log = LogFactory.getLog(FeedMissingClaimsJobNode.class);
    public static final int BATCH_SIZE = 1000;
    public static final int ATTEMPTS = 3;
    public static final int SLEEP_MS_SOLR_CLIENT = 5000;
    private RecentResultsQueue queue;
    private OafToIndexRecordFactory oafToIndexRecordFactory;

    @Resource
    private UniqueServiceLocator serviceLocator;

    @Value(value = "${openaire.api.directindex.findSolrIndexUrl.xquery}")
    private ClassPathResource findSolrIndexUrl;

    @Override
    protected String execute(final NodeToken nodeToken) throws Exception {

        final String format =
                nodeToken.getEnv().hasAttribute("format") ? nodeToken.getEnv().getAttribute("format") : nodeToken.getFullEnv().getAttribute("format");
        final String coll = format + "-index-openaire";
        final String baseUrl = calculateIndexBaseUrl();

        CloudIndexClient idxClient = null;

        try {
            final List<SolrInputDocument> toFeed = new ArrayList<SolrInputDocument>();
            final List<String> toDeleteFromCache = new ArrayList<String>();

            final SAXReader reader = new SAXReader();
            final ApplyXslt xslt = oafToIndexRecordFactory.newTransformer(format);

            idxClient = CloudIndexClientFactory.newIndexClient(baseUrl, coll, false);
            log.info("Starting to feed claims in index collection " + coll);
            int count = 0;
            for (String record : queue) {
                int max_attempts = ATTEMPTS;
                count++;
                final String id = reader.read(new StringReader(record)).valueOf("//*[local-name() = 'objIdentifier']");
                if (log.isDebugEnabled()) {
                    log.debug(String.format("Processing record %s, number: %d", id, count));
                }
                if (isRecordIndexed(idxClient, baseUrl, coll, id, max_attempts)) {
                    toDeleteFromCache.add(id);
                } else {
                    max_attempts = ATTEMPTS;
                    toFeed.add(prepareSolrDoc(idxClient, baseUrl, coll, id, record, xslt, max_attempts));
                }
                if (count % BATCH_SIZE == 0) processLists(idxClient, baseUrl, coll, toFeed, toDeleteFromCache);
            }
            if (!toFeed.isEmpty() || !toDeleteFromCache.isEmpty()) processLists(idxClient, baseUrl, coll, toFeed, toDeleteFromCache);
            log.info(String.format("Finished feeding of claims in index collection %s, total: %d", coll, count));

        } catch (Throwable e) {
            log.error("Error feeding missing claims", e);
            throw e;
        } finally {
            if (idxClient != null) {
                idxClient.close();
            }
            log.info("Closed Solr index client");
        }
        log.info("Now proceeding to Arc.DEFAULT_ARC");
        return Arc.DEFAULT_ARC;
    }

    protected boolean isRecordIndexed(CloudIndexClient idxClient, String baseUrl, String coll, String id, int attempt) throws IOException, CloudIndexClientException, MSROException, InterruptedException {
        try {
            return idxClient.isRecordIndexed(id);
        } catch (CloudIndexClientException cie) {
            log.error(String.format("Error querying for %s, message: %s. Trying again, remaining attempts:", id, cie, attempt));
            idxClient = resetCloudIndexClient(idxClient, baseUrl, coll);
            if (attempt > 0) return isRecordIndexed(idxClient, baseUrl, coll, id, --attempt);
            else {
                String msg = String.format("Too many attempts %d to recreate the index client for checking if record %s exists.", ATTEMPTS, id);
                log.error(msg);
                throw new MSROException(cie);
            }
        }
    }

    protected SolrInputDocument prepareSolrDoc(CloudIndexClient idxClient, String baseUrl, String coll, String recordId, String record, ApplyXslt xslt, int attempt) throws IOException, CloudIndexClientException, MSROException, InterruptedException {
        try {
            return idxClient.prepareSolrDocument(record, xslt);
        } catch (CloudIndexClientException cie) {
            log.error(String.format("Error preparing Solr doc for %s, message: %s. Trying again, remaining attempts:", recordId, cie, attempt));
            idxClient = resetCloudIndexClient(idxClient, baseUrl, coll);
            if (attempt > 0)
                return prepareSolrDoc(idxClient, baseUrl, coll, recordId, record, xslt, --attempt);
            else {
                String msg = String.format("Too many attempts %d to recreate the index client for preparing SolrDocument for %s", ATTEMPTS, id);
                log.error(msg);
                throw new MSROException(cie);
            }
        }
    }

    protected void tryToFeed(CloudIndexClient idxClient, String baseUrl, String coll, List<SolrInputDocument> toFeed, int attempt) throws MSROException, IOException, CloudIndexClientException, InterruptedException {
        try {
            idxClient.feed(toFeed, null);
        } catch (CloudIndexClientException cie) {
            log.error(String.format("Error feeding Solr in attempt number %d", attempt));
            idxClient = resetCloudIndexClient(idxClient, baseUrl, coll);
            if (attempt > 0) tryToFeed(idxClient, baseUrl, coll, toFeed, --attempt);
            else {
                String msg = String.format("Too many attempts %d to recreate the index client for feeding Solr", ATTEMPTS);
                log.error(msg);
                throw new MSROException(cie);
            }
        }
    }

    private CloudIndexClient resetCloudIndexClient(CloudIndexClient idxClient, String baseUrl, String coll) throws IOException, CloudIndexClientException, InterruptedException {
        if (idxClient != null) {
            idxClient.close();
        }
        Thread.sleep(SLEEP_MS_SOLR_CLIENT);
        CloudIndexClient newclient = CloudIndexClientFactory.newIndexClient(baseUrl, coll, false);
        log.info("Got new CloudIndexClient");
        return newclient;
    }


    private void processLists(final CloudIndexClient idxClient, String baseUrl, String coll, final List<SolrInputDocument> toFeed, final List<String> toDeleteFromCache) throws CloudIndexClientException, MSROException, IOException, InterruptedException {
        int max_attempts = ATTEMPTS;
        tryToFeed(idxClient, baseUrl, coll, toFeed, max_attempts);
        queue.remove(toDeleteFromCache);
        log.info(String.format("%d claims fed and cache cleaned of %d records", toFeed.size(), toDeleteFromCache.size()));
        toFeed.clear();
        toDeleteFromCache.clear();
        log.info("Cleaned temporary lists");
    }

    public RecentResultsQueue getQueue() {
        return queue;
    }

    @Required
    public void setQueue(final RecentResultsQueue queue) {
        this.queue = queue;
    }

    public OafToIndexRecordFactory getOafToIndexRecordFactory() {
        return oafToIndexRecordFactory;
    }

    @Required
    public void setOafToIndexRecordFactory(final OafToIndexRecordFactory oafToIndexRecordFactory) {
        this.oafToIndexRecordFactory = oafToIndexRecordFactory;
    }

    private String calculateIndexBaseUrl() throws Exception {
        final String query = IOUtils.toString(findSolrIndexUrl.getInputStream());
        return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query);
    }
}
