package eu.dnetlib.msro.openaireplus.workflows.nodes.contexts;

import java.io.StringReader;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.Map.Entry;

import com.google.common.base.Function;
import com.google.common.collect.Maps;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.*;
import org.dom4j.io.SAXReader;

/**
 * Created by claudio on 01/03/16.
 */
public class ContextUtils {

	private static final Log log = LogFactory.getLog(ContextUtils.class);
	private static final String DASHBOARD_VISIBILITY = "status";

	public static ContextDesc getContext(final Iterable<String> it,
			final String contextId,
			final String contextLabel,
			final String contextType,
			final String params, final String dashboardVisibility) throws
			DocumentException {

		final Map<String, String> paramMap = getParamMap(params);
		paramMap.put(DASHBOARD_VISIBILITY, dashboardVisibility);
		return new ContextUtils().getContextDesc(it, contextId, contextLabel, contextType, paramMap);
	}

	private static Map<String, String> getParamMap(String jsonMap) {
		Map<String, String> paramMap = Maps.newHashMap();

		if (StringUtils.isNotBlank(jsonMap)) {
			Type mapType = new TypeToken<Map<String, String>>() {
			}.getType();
			paramMap = new Gson().fromJson(jsonMap, mapType);
		}
		return paramMap;
	}

	private ContextDesc getContextDesc(final Iterable<String> it,
			final String contextId,
			final String contextLabel,
			final String contextType,
			final Map<String, String> params) throws
			DocumentException {
		final ContextDesc context = new ContextDesc(contextId, contextLabel, contextType, params);
		final SAXReader reader = new SAXReader();

		for (String s : it) {
			populateContext(context, reader, s);
		}
		return context;
	}

	private void populateContext(final ContextDesc context, final SAXReader reader, final String s) throws DocumentException {
		final Document doc = reader.read(new StringReader(s));

		for (Object o : doc.selectNodes("//fundingtree")) {
			final Element treeNode = (Element) o;

			final String funder = treeNode.valueOf("./funder/id");
			if (StringUtils.isBlank(funder)) {
				log.debug("No funder found, skipping population from the following XML: \n" + s);
				return;
			}
			if (!context.getDbEntries().containsKey(funder)) {
				log.info("Found funder: " + funder);
				context.getDbEntries().put(funder, "<fundingtree>" + treeNode.selectSingleNode("./funder").asXML() + "</fundingtree>");
				log.debug("db entry: " + context.getDbEntries().get(funder));
			}

			final String openaireId = treeNode.valueOf("./*[starts-with(local-name(),'funding_level_')]/id");
			if (!context.getDbEntries().containsKey(openaireId) && StringUtils.isNotBlank(openaireId)) {
				log.info("Found funding: " + openaireId);
				context.getDbEntries().put(openaireId, treeNode.asXML());
				log.debug("db entry: " + context.getDbEntries().get(openaireId));
				final Node node0 = treeNode.selectSingleNode(".//funding_level_0");
				if (node0 != null) {
					final ContextPart part = calculatePart(node0);
					if (context.getCategories().containsKey(part.getId())) {
						for (ContextPart p : part.getParts().values()) {
							context.getCategories().get(part.getId()).addPart(p);
						}
					} else {
						context.getCategories().put(part.getId(), part);
					}
				}
			}
		}
	}

	private ContextPart calculatePart(final Node node) {

		// final String newId = contextId + "::" + StringUtils.substringAfter(node.valueOf("./id"), "::");
		// ids are built as: nsPrefix :: funderID :: fundingLevel0ID :: etc etc, hence it seems we might not need the contextId parameter.
		final String newId = StringUtils.substringAfter(node.valueOf("./id"), "::");

		final ContextPart part = new ContextPart(newId, node.valueOf("./description"));

		part.getParams().put("name", node.valueOf("./name"));
		part.getParams().put("openaireId", node.valueOf("./id"));
		part.getParams().put("class", node.valueOf("./class"));

		final Element parent = node.getParent() != null ? node.getParent().getParent() : null;

		if (parent != null && parent.getName().startsWith("funding_level_")) {
			final ContextPart p = calculatePart(parent);
			part.getParts().put(p.getId(), p);
		}
		return part;
	}

	public static Function<Entry<String, String>, String> getContextRowTransformer() {

		return new Function<Entry<String, String>, String>() {

			private final SAXReader reader = new SAXReader();

			@Override
			public String apply(final Entry<String, String> e) {
				try {
					final Document docFundingPath = reader.read(new StringReader(e.getValue()));

					final Element root = DocumentHelper.createElement("ROWS");

					final Map<String, String> fundingFields = Maps.newLinkedHashMap();
					fundingFields.put("_dnet_resource_identifier_", e.getKey());
					fundingFields.put("id", e.getKey());
					fundingFields.put("path", e.getValue());
					fundingFields.put("jurisdiction", docFundingPath.valueOf("//funder/jurisdiction"));
					final String desc = docFundingPath.valueOf("(//*[starts-with(local-name(),'funding_level_')])[1]/description");
					if (StringUtils.isNotBlank(desc)) {
						fundingFields.put("description", desc);
					}
					final Map<String, String> orgFields = findFunderInfo(docFundingPath);
					final String orgId = orgFields.get("id");
					if (StringUtils.isNotBlank(orgId)) {
						addRow(root, "dsm_organizations", orgFields);
						fundingFields.put("funder", orgId);
					}

					addRow(root, "fundingpaths", fundingFields);

					if (log.isDebugEnabled()) {
						log.debug("Db entries: " + root.asXML());
					}

					return root.asXML();
				} catch (DocumentException e1) {
					log.error("Error parsing xml", e1);
					throw new RuntimeException("Error parsing xml", e1);
				}
			}

			private Map<String, String> findFunderInfo(final Document doc) {
				final Map<String, String> res = Maps.newLinkedHashMap();
				res.put("_dnet_resource_identifier_", doc.valueOf("//funder/id"));
				res.put("id", doc.valueOf("//funder/id"));
				res.put("legalshortname", doc.valueOf("//funder/shortname"));
				res.put("legalname", doc.valueOf("//funder/name"));
				res.put("country", doc.valueOf("//funder/jurisdiction"));
				return res;
			}

			private void addRow(final Element root, final String table, final Map<String, String> fields) {
				final Element row = root.addElement("ROW");
				row.addAttribute("table", table);
				for (Map.Entry<String, String> e : fields.entrySet()) {
					final Element pathField = row.addElement("FIELD");
					pathField.addAttribute("name", e.getKey());
					pathField.setText(e.getValue());
				}
			}
		};
	}

}
