package eu.dnetlib.ariadneplus.publisher;

import java.io.IOException;
import java.net.URISyntaxException;

import eu.dnetlib.ariadneplus.jrr.JRRPublisher;
import eu.dnetlib.ariadneplus.virtuoso.VirtuosoClient;
import eu.dnetlib.ariadneplus.virtuoso.VirtuosoClientFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Created by Alessia Bardi on 11/08/2017.
 *
 * @author Alessia Bardi
 */
@Component
public class AriadnePlusPublisherHelper {

	private static final Log log = LogFactory.getLog(AriadnePlusPublisherHelper.class);

	public enum AriadnePlusTargets{
		VIRTUOSO, JRR
	}

	@Autowired
	private VirtuosoClientFactory virtuosoClientFactory;
	@Autowired
	private JRRPublisher jrrPublisher;

	public void publish(final String record, final AriadnePlusTargets target) throws AriadnePlusPublisherException {
		switch(target){
		case VIRTUOSO:
			publishVirtuoso(record);
			break;
		case JRR:
			try {
				publishJRR(record);
				break;
			} catch(IOException | URISyntaxException | InterruptedException e){
				throw new AriadnePlusPublisherException(e);
			}
		default: throw new AriadnePlusPublisherException("Target "+target+" not supported yet");
		}

	}

	public long unpublish(final String datasourceInterface, final AriadnePlusTargets target) throws AriadnePlusPublisherException {
		long res = 0;
		switch(target){
		case VIRTUOSO:
			res = unpublishVirtuoso(datasourceInterface);
			break;
		default: throw new AriadnePlusPublisherException("Target "+target+" not supported yet");
		}
		return res;
	}

	private void publishVirtuoso(final String record) throws AriadnePlusPublisherException {
		log.debug("Publishing on virtuoso");
		VirtuosoClient virtuosoClient = this.virtuosoClientFactory.getVirtuosoClient();
		virtuosoClient.feed(record);
	}

	private void publishJRR(final String record)
			throws AriadnePlusPublisherException, IOException, URISyntaxException, InterruptedException {
		log.debug("Publishing on JRR (registry and catalogue)");
		jrrPublisher.register(record);
	}

	private int unpublishJRR(final String datasourceInterface){
		//TODO: for this to work we have to add somewhere the information about the dsInterface from which the resource was initially collected
		//Note that this method might not be a good idea if we want to keep the uuid and only update the facets/rels
		//maybe it is worth to implement the incremental in the ResourceRegistrator. We slow down things, but it may be worthy...
		log.debug("Unpublishing from registry "+datasourceInterface);
		//TODO: implement me
		throw new UnsupportedOperationException("Not implemented yet");
	}

	private long unpublishVirtuoso(final String datasourceInterface) {
		log.info("Unpublishing from virtuoso "+datasourceInterface);
		VirtuosoClient virtuosoClient = this.virtuosoClientFactory.getVirtuosoClient();
		long deletedTriples =  virtuosoClient.drop(datasourceInterface);
		log.info("# triples deleted for "+datasourceInterface+": "+deletedTriples);
		return deletedTriples;
	}

	public void dropRegistry(){
		log.debug("Dropping JRR");
		//TODO: implement me
		throw new UnsupportedOperationException("Not implemented yet");
	}


}
