package eu.dnetlib.miscutils.functional.xml;

import java.io.IOException;
import java.io.StringReader;
import java.util.function.UnaryOperator;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;



/**
 * The Class XMLIndenter.
 */
public class XMLIndenter implements UnaryOperator<String> {

	private static final Log log = LogFactory.getLog(XMLIndenter.class);

	/**
	 * Static helper Apply.
	 *
	 * @param xml the xml
	 * @return the indented xml string
	 */
	public static String indent(final String xml) {
		return new XMLIndenter().apply(xml);
	}

	public static String indent(final Document document) throws Exception {
		return new XMLIndenter().doIndent(document);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.miscutils.functional.UnaryFunction#evaluate(java.lang.Object)
	 */
	@Override
	public String apply(final String unformattedXml) {
		try {
			return doIndent(unformattedXml);
		} catch (Throwable e) {
			return unformattedXml;
		}
	}

	/**
	 * Do indent.
	 *
	 * @param unformattedXml
	 *            the unformatted xml
	 * @return the string
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	protected String doIndent(final String unformattedXml) throws Exception {
		return doIndent(parseXmlString(unformattedXml));
	}

	private String doIndent(final Document document) throws Exception {

		TransformerFactory tf = TransformerFactory.newInstance();
		Transformer transformer = tf.newTransformer();
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");
		transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, "yes");
		transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
		final ByteArrayOutputStream out = new ByteArrayOutputStream();
		transformer.transform(new DOMSource(document), new StreamResult(out));

		return out.toString("utf-8");
	}

	/**
	 * Parses the xml string.
	 *
	 * @param in
	 *            the in
	 * @return the document
	 */
	public Document parseXmlString(final String in) {
		try {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			DocumentBuilder db = dbf.newDocumentBuilder();
			db.setErrorHandler(null);
			InputSource is = new InputSource(new StringReader(in));
			return db.parse(is);
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		} catch (SAXException e) {
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

}
