package eu.dnetlib.clients.enabling;

import java.util.*;
import java.util.stream.Collectors;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.PropertyAccessorFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Created by sandro on 3/21/16.
 */
public class ISLookUpClient {

	private static final Log log = LogFactory.getLog(ISLookUpClient.class);

	@Autowired
	private UniqueServiceLocator serviceLocator;

	private ISLookUpService lookUpService;

	public void initialize() {
		log.debug("Initializing the ISLookUpService");
		setLookUpService(serviceLocator.getService(ISLookUpService.class));
	}

	public String getResourceProfileById(final String id) throws ISLookUpException {
		return getLookUpService().getResourceProfile(id);
	}

	public String getResourceProfileByQuery(final String xquery) throws ISLookUpException {
		return getLookUpService().getResourceProfileByQuery(xquery);
	}

	public List<String> search(final String xquery) throws ISLookUpException {
		return getLookUpService().quickSearchProfile(xquery);
	}

	/**
	 * This method return a List of object of generic class
	 * obtained by splitting the query using the param separator.
	 * <p>
	 * Each query MUST return a sequence like key1 separator value1 separator key2 separator value2 ....
	 * Each key should match a name of the attribute of the class
	 *
	 * @param query
	 * @param clazz
	 * @param separator
	 * @return
	 * @throws ISLookUpException
	 */
	public <T> List<T> search(final String query, final Class<T> clazz, final String separator) throws ISLookUpException {
		try {
			return getLookUpService().quickSearchProfile(query)
					.stream()
					.map(it -> convertResult(it, clazz, separator))
					.collect(Collectors.toList());
		} catch (Throwable e) {
			throw new ISLookUpException(e);
		}
	}

	/**
	 * This method return a List of object of generic class
	 * obtained by splitting the query using the param separator.
	 * <p>
	 * Each query MUST return a sequence values1 separator value1 separator value3 ....
	 *
	 * @param query
	 * @param clazz
	 * @param separator
	 * @return
	 * @throws ISLookUpException
	 */
	public <T> List<T> searchAndMapToClassByConstructor(final String query, final Class<T> clazz, final String separator) throws ISLookUpException {
		final List<String> results = getLookUpService().quickSearchProfile(query);
		try {
			return results.stream().map(result -> constructObjectByResult(clazz, separator, result)).collect(Collectors.toList());
		} catch (Throwable e) {
			throw new ISLookUpException(e);
		}
	}

	public <T> T constructObjectByResult(final Class<T> clazz, final String separator, final String result) {
		try {
			if (result == null)
				throw new RuntimeException("the result of query is null");
			List<String> splittedValues = Arrays.asList(result.split(separator));
			if (StringUtils.endsWith(result, separator)) {
				splittedValues.add("");
			}
			List<Class> types = new ArrayList<>();
			splittedValues.forEach(it -> types.add(String.class));

			return clazz.getDeclaredConstructor(types.toArray(new Class[] {})).newInstance(splittedValues.toArray(new String[] {}));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private <T> T convertResult(final String result, final Class<T> clazz, final String separator) {
		if (result == null)
			throw new RuntimeException("the result of query is null");

		List<String> splittedValues = Arrays.asList(result.split(separator));
		if (StringUtils.endsWith(result, separator)) {
			splittedValues.add("");
		}
		if (splittedValues.size() % 2 != 0)
			throw new RuntimeException("the number of argument of result must be key" + separator + "value in  " + result);
		try {
			Map<String, String> keyValuesMap = new HashMap<>();
			for (int i = 0; i < splittedValues.size() - 1; i += 2) {
				keyValuesMap.put(splittedValues.get(i), splittedValues.get(i + 1));
			}
			T object = clazz.newInstance();
			PropertyAccessorFactory.forBeanPropertyAccess(object).setPropertyValues(keyValuesMap);
			return object;
		} catch (Exception e) {
			throw new RuntimeException("Error on creating object of class: " + clazz.getCanonicalName());
		}
	}

	public ISLookUpService getLookUpService() {
		if (lookUpService == null)
			initialize();
		return lookUpService;
	}

	public void setLookUpService(final ISLookUpService lookUpService) {
		this.lookUpService = lookUpService;
	}

}
