package eu.dnetlib.rmi.data.hadoop;

import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.jws.WebParam;
import javax.jws.WebService;

import eu.dnetlib.rmi.common.BaseService;
import eu.dnetlib.rmi.data.hadoop.hbase.HBaseRowDescriptor;

/**
 * The Interface HadoopService.
 */
@WebService(targetNamespace = "http://services.dnetlib.eu/")
public interface HadoopService extends BaseService {

	/**
	 * List available jobs.
	 *
	 * @return the list
	 * @throws HadoopServiceException the hadoop service exception
	 */
	List<String> listAvailableJobs() throws HadoopServiceException;

	/**
	 * List jobs.
	 *
	 * @param clusterName the cluster name
	 * @return the list
	 * @throws HadoopServiceException the hadoop service exception
	 */
	List<HadoopJobDescriptor> listJobs(@WebParam(name = "clusterName") String clusterName) throws HadoopServiceException;

	/**
	 * Kill job.
	 *
	 * @param jobId the job id
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean killJob(@WebParam(name = "id") String jobId) throws HadoopServiceException;

	/**
	 * List hbase tables.
	 *
	 * @param clusterName the cluster name
	 * @return the list
	 * @throws HadoopServiceException the hadoop service exception
	 */
	List<String> listHbaseTables(@WebParam(name = "clusterName") String clusterName) throws HadoopServiceException;

	/**
	 * Ensure hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @param columns     the columns
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean ensureHbaseTable(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "columns") Set<String> columns) throws HadoopServiceException;

	/**
	 * Creates the hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @param columns     the columns
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean createHbaseTable(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "columns") Set<String> columns) throws HadoopServiceException;

	/**
	 * Creates the hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @param tableConfiguration     the table configuration: columns and region splits
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean createConfiguredHbaseTable(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "tableConfiguration") String tableConfiguration) throws HadoopServiceException;

	/**
	 * Truncate hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean truncateHbaseTable(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "tableName") String tableName)
			throws HadoopServiceException;

	/**
	 * Drop hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean dropHbaseTable(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "tableName") String tableName) throws HadoopServiceException;

	/**
	 * Exist hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @return true, if successful
	 * @throws HadoopServiceException the hadoop service exception
	 */
	boolean existHbaseTable(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "tableName") String tableName) throws HadoopServiceException;

	/**
	 * Describe hbase table.
	 *
	 * @param clusterName the cluster name
	 * @param tableName   the table name
	 * @return the list of columns defined for that table.
	 * @throws HadoopServiceException the hadoop service exception
	 */
	List<String> describeHbaseTable(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "tableName") String tableName)
			throws HadoopServiceException;

	/**
	 * Gets the cluster configuration.
	 *
	 * @param clusterName the cluster name
	 * @return the cluster configuration
	 * @throws HadoopServiceException the hadoop service exception
	 */
	Map<String, String> getClusterConfiguration(@WebParam(name = "clusterName") String clusterName) throws HadoopServiceException;

	/**
	 * Deletes a path on HDFS
	 *
	 * @param clusterName the cluster name
	 * @param path        the path to be deleted
	 * @return true in case of success
	 * @throws HadoopServiceException
	 */
	boolean deleteHdfsPath(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "path") String path)
			throws HadoopServiceException;

	/**
	 * Check if path on HDFS exists
	 *
	 * @param clusterName the cluster name
	 * @param path        the path to be deleted
	 * @return true in case the path exists
	 * @throws HadoopServiceException
	 */
	boolean existHdfsPath(@WebParam(name = "clusterName") String clusterName, @WebParam(name = "path") String path)
			throws HadoopServiceException;

	/**
	 * Creates a directory on HDFS, eventually deleteing its content.
	 *
	 * @param clusterName the cluster name
	 * @param path        the path to be created
	 * @param force       force delete before create
	 * @return true in case of success
	 * @throws HadoopServiceException
	 */
	boolean createHdfsDirectory(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "path") String path,
			@WebParam(name = "force") boolean force)
			throws HadoopServiceException;

	/**
	 * List the configured cluster names.
	 *
	 * @return the list of configured cluster names.
	 * @throws HadoopServiceException shouldn't happen
	 */
	List<String> listClusters() throws HadoopServiceException;

	/**
	 * Get the columns defined in a given row
	 *
	 * @param clusterName
	 * @param tableName
	 * @param rowKey
	 * @return
	 */
	HBaseRowDescriptor describeHBaseColumn(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "rowKey") String rowKey) throws HadoopServiceException;

	/**
	 * Get the columns defined in a given set of row
	 *
	 * @param clusterName
	 * @param tableName
	 * @param rowKeys
	 * @return
	 */
	Map<String, HBaseRowDescriptor> describeHBaseColumns(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "rowKeys") List<String> rowKeys) throws HadoopServiceException;

	/**
	 * Get the HBase table schema and the region splits configuration
	 *
	 * @param clusterName
	 * @param tableName
	 * @return
	 */
	String describeHBaseTableConfiguration(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName) throws HadoopServiceException;

	/**
	 * Remove from an HBase table the values identified by a row descriptor.
	 *
	 * @param clusterName
	 * @param tableName
	 * @param column
	 * @return
	 */
	boolean deleteHBaseColumn(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "column") HBaseRowDescriptor column) throws HadoopServiceException;

	/**
	 * Remove from an HBase table the values identified by a row descriptor.
	 *
	 * @param clusterName
	 * @param tableName
	 * @param column
	 * @return
	 */
	boolean deleteHBaseColumns(@WebParam(name = "clusterName") String clusterName,
			@WebParam(name = "tableName") String tableName,
			@WebParam(name = "column") List<HBaseRowDescriptor> column) throws HadoopServiceException;

}
