package eu.dnetlib.clients.index.model.document;

import java.util.*;

import eu.dnetlib.clients.index.model.Any.ValueType;
import eu.dnetlib.clients.index.utils.IndexFieldUtility;

/**
 * The Class AbstractDocument Represent the field information needed to construct and index Document.
 */
public abstract class AbstractIndexDocument implements Map<String, IndexField>, Iterable<IndexField>, IndexDocument {

	private static final String DATE_SUFFIX = "\\+\\d{2}:\\d{2}";
	/**
	 * The _fields.
	 */
	protected final Map<String, IndexField> fields;
	/**
	 * index schema.
	 */
	protected final Map<String, ValueType> schema;
	/**
	 * Actual document status.
	 */
	private Status status;
	/**
	 * If there was an error building the document, it is described here.
	 */
	private Throwable error;

	/**
	 * Instantiates a new abstract document.
	 */
	public AbstractIndexDocument(final Map<String, ValueType> schema, final String dsId) {
		this.schema = schema;
		fields = new LinkedHashMap<>();
	}

	/**
	 * Instantiates a new abstract document.
	 *
	 * @param fields the fields
	 */
	public AbstractIndexDocument(final Map<String, ValueType> schema, final String dsId, final Map<String, IndexField> fields) {
		this.schema = schema;
		this.fields = fields;
		addField(IndexFieldUtility.DS_ID, dsId);
	}

	/**
	 * Remove all fields and boosts from the document.
	 */
	@Override
	public void clear() {
		if (fields != null) {
			fields.clear();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addField(final String name, final Object value) {
		Object hack_value = hackField(name, value);
		IndexField field = fields.get(name);
		if ((field == null) || (field.getValue() == null)) {
			setField(name, hack_value);
		} else {
			field.addValue(hack_value);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getFieldValue(final String name) {
		IndexField field = getField(name);
		Object o = null;
		if (field != null) {
			o = field.getFirstValue();
		}
		return o;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexField getField(final String field) {
		return fields.get(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<Object> getFieldValues(final String name) {
		IndexField field = getField(name);
		if (field != null) return field.getValues();
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<String> getFieldNames() {
		return fields.keySet();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setField(final String name, final Object value) {
		Object hack_value = hackField(name, value);
		IndexField field = new IndexField(name);
		fields.put(name, field);
		field.setValue(hack_value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexField removeField(final String name) {
		return fields.remove(name);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Iterable#iterator()
	 */
	@Override
	public Iterator<IndexField> iterator() {
		return fields.values().iterator();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Object#toString()
	 */
	@Override
	public String toString() {
		return "IndexDocument: " + fields.values();
	}

	// ---------------------------------------------------
	// MAP interface
	// ---------------------------------------------------

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#containsKey(Object)
	 */
	@Override
	public boolean containsKey(final Object key) {
		return fields.containsKey(key);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#containsValue(Object)
	 */
	@Override
	public boolean containsValue(final Object value) {
		return fields.containsValue(value);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#entrySet()
	 */
	@Override
	public Set<Entry<String, IndexField>> entrySet() {
		return fields.entrySet();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#get(Object)
	 */
	@Override
	public IndexField get(final Object key) {
		return fields.get(key);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#isEmpty()
	 */
	@Override
	public boolean isEmpty() {
		return fields.isEmpty();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#keySet()
	 */
	@Override
	public Set<String> keySet() {
		return fields.keySet();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#put(Object, Object)
	 */
	@Override
	public IndexField put(final String key, final IndexField value) {
		return fields.put(key, value);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#putAll(Map)
	 */
	@Override
	public void putAll(final Map<? extends String, ? extends IndexField> t) {
		fields.putAll(t);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#remove(Object)
	 */
	@Override
	public IndexField remove(final Object key) {
		return fields.remove(key);
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#size()
	 */
	@Override
	public int size() {
		return fields.size();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see Map#values()
	 */
	@Override
	public Collection<IndexField> values() {
		return fields.values();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Status getStatus() {
		return status;
	}

	/**
	 * Sets the status.
	 *
	 * @param status the status to set
	 * @return the index document
	 */
	@Override
	public IndexDocument setStatus(final Status status) {
		this.status = status;
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexDocument deepCopy() {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Throwable getError() {

		return error;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexDocument setOK() {

		return this.setStatus(Status.OK);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexDocument setMarked() {
		addField(IndexFieldUtility.DELETE_DOCUMENT, true);
		return this.setStatus(Status.MARKED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IndexDocument setError(final Throwable error) {
		this.error = error;
		return this.setStatus(Status.ERROR);
	}

	/**
	 * This method modifies a field w.r.t. his type
	 *
	 * @param name
	 * @param value
	 * @return
	 */
	private Object hackField(final String name, Object value) {
		if (schema == null) return value;

		final ValueType type = schema.get(name);

		// TODO: hack this is hardcoded for the date type
		switch (type) {
		case DATETIME:
		case DATE:
			if (value.toString().matches(DATE_SUFFIX)) return value.toString().replaceAll(DATE_SUFFIX, "Z");
			if (value.toString().endsWith("Z")) return value;

			value = value + "T00:00:00Z";
			break;
		default:
			break;
		}
		return value;
	}

}
