package eu.dnetlib.rmi.provision;

import eu.dnetlib.utils.MetadataReference;

public class MDFInfo {

	/**
	 * prefix, schema and namespace are properties describing the metadata format exported via OAI PMH.
	 */
	private String prefix;
	private String schema;
	private String namespace;

	/**
	 * sourceFormat* are properties describing the metadata format to use as input to deliver records in the <prefix> format.
	 * sourceFormatName may be the same as prefix, for example in cases when we have a oai_dc store and we want to deliver oai_dc records.
	 */
	private MetadataReference sourceMetadataReference;

	/**
	 * query to filter out records to be delivered.
	 */
	private String baseQuery;
	/**
	 * identificator of the TDS profile containing an XSLT to be applied to records of <sourceFormatName> before delivering. The
	 * transformation transforms them into the <prefix> metadata format. This property can be null/empty in cases when the delivered format
	 * is the same of the source format.
	 */
	private String transformationRuleID;

	/**
	 * Tells if the metadata format is enabled for export. If not, it won't appear in the ListMetadataFormats response.
	 */
	private boolean enabled;

	public MDFInfo() {
	}

	public MDFInfo(final String prefix, final String schema, final String namespace, final String sourceFormatName, final String sourceFormatLayout,
			final String sourceFormatInterpretation, final String baseQuery, final boolean enabled) {
		super();
		initCommonFields(prefix, schema, namespace, sourceFormatName, sourceFormatLayout, sourceFormatInterpretation, baseQuery);
		this.enabled = enabled;
	}

	/**
	 * This constructor is the same as the above, but in this case "enabled" is a string, so we can use the SplittedQueryExecutor to
	 * generate instances of this class from the information stored in the D-Net IS.
	 *
	 * @param prefix
	 * @param schema
	 * @param namespace
	 * @param sourceFormatName
	 * @param sourceFormatLayout
	 * @param sourceFormatInterpretation
	 * @param baseQuery
	 * @param enabled
	 */
	public MDFInfo(final String prefix, final String schema, final String namespace, final String sourceFormatName, final String sourceFormatLayout,
			final String sourceFormatInterpretation, final String baseQuery, final String enabled) {
		super();
		initCommonFields(prefix, schema, namespace, sourceFormatName, sourceFormatLayout, sourceFormatInterpretation, baseQuery);
		this.enabled = Boolean.valueOf(enabled);
	}

	public MDFInfo(final String prefix, final String schema, final String namespace, final String sourceFormatName, final String sourceFormatLayout,
			final String sourceFormatInterpretation, final String baseQuery, final String tdsID, final boolean enabled) {
		this(prefix, schema, namespace, sourceFormatName, sourceFormatLayout, sourceFormatInterpretation, baseQuery, enabled);
		transformationRuleID = tdsID;
	}

	/**
	 * This constructor is the same as the above, but in this case "enabled" is a string, so we can use the SplittedQueryExecutor to
	 * generate instances of this class from the information stored in the D-Net IS.
	 *
	 * @param prefix
	 * @param schema
	 * @param namespace
	 * @param sourceFormatName
	 * @param sourceFormatLayout
	 * @param sourceFormatInterpretation
	 * @param baseQuery
	 * @param tdsID
	 * @param enabled
	 */
	public MDFInfo(final String prefix, final String schema, final String namespace, final String sourceFormatName, final String sourceFormatLayout,
			final String sourceFormatInterpretation, final String baseQuery, final String tdsID, final String enabled) {
		this(prefix, schema, namespace, sourceFormatName, sourceFormatLayout, sourceFormatInterpretation, baseQuery, enabled);
		transformationRuleID = tdsID;
	}

	private void initCommonFields(final String prefix,
			final String schema,
			final String namespace,
			final String sourceFormatName,
			final String sourceFormatLayout,
			final String sourceFormatInterpretation,
			final String baseQuery) {
		this.prefix = prefix;
		this.schema = schema;
		this.namespace = namespace;
		this.sourceMetadataReference = new MetadataReference(sourceFormatName, sourceFormatLayout, sourceFormatInterpretation);
		this.baseQuery = baseQuery;
	}

	@Override
	public String toString() {
		return "MDFInfo{" +
				"prefix='" + prefix + '\'' +
				", schema='" + schema + '\'' +
				", namespace='" + namespace + '\'' +
				", sourceMetadataReference=" + sourceMetadataReference +
				", baseQuery='" + baseQuery + '\'' +
				", transformationRuleID='" + transformationRuleID + '\'' +
				", enabled=" + enabled +
				'}';
	}

	public String getPrefix() {
		return prefix;
	}

	public void setPrefix(final String prefix) {
		this.prefix = prefix;
	}

	public String getSchema() {
		return schema;
	}

	public void setSchema(final String schema) {
		this.schema = schema;
	}

	public String getNamespace() {
		return namespace;
	}

	public void setNamespace(final String namespace) {
		this.namespace = namespace;
	}

	public String getBaseQuery() {
		return baseQuery;
	}

	public void setBaseQuery(final String baseQuery) {
		this.baseQuery = baseQuery;
	}

	public String getTransformationRuleID() {
		return transformationRuleID;
	}

	public void setTransformationRuleID(final String transformationRuleID) {
		this.transformationRuleID = transformationRuleID;
	}

	public boolean isEnabled() {
		return enabled;
	}

	public void setEnabled(final boolean enabled) {
		this.enabled = enabled;
	}

	public MetadataReference getSourceMetadataReference() {
		return sourceMetadataReference;
	}

	public void setSourceMetadataReference(final MetadataReference sourceMetadataReference) {
		this.sourceMetadataReference = sourceMetadataReference;
	}

	public String getSourceFormat(){
		return getSourceMetadataReference().getFormat();
	}

	public String getSourceLayout(){
		return getSourceMetadataReference().getLayout();
	}
	public String getSourceInterpretation(){
		return getSourceMetadataReference().getInterpretation();
	}


	@Override
	public boolean equals(final Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		final MDFInfo mdfInfo = (MDFInfo) o;

		if (isEnabled() != mdfInfo.isEnabled()) return false;
		if (getPrefix() != null ? !getPrefix().equals(mdfInfo.getPrefix()) : mdfInfo.getPrefix() != null) return false;
		if (getSchema() != null ? !getSchema().equals(mdfInfo.getSchema()) : mdfInfo.getSchema() != null) return false;
		if (getNamespace() != null ? !getNamespace().equals(mdfInfo.getNamespace()) : mdfInfo.getNamespace() != null) return false;
		if (getSourceMetadataReference() != null ?
				!getSourceMetadataReference().equals(mdfInfo.getSourceMetadataReference()) :
				mdfInfo.getSourceMetadataReference() != null)
			return false;
		if (getBaseQuery() != null ? !getBaseQuery().equals(mdfInfo.getBaseQuery()) : mdfInfo.getBaseQuery() != null) return false;
		return getTransformationRuleID() != null ?
				getTransformationRuleID().equals(mdfInfo.getTransformationRuleID()) :
				mdfInfo.getTransformationRuleID() == null;
	}

	@Override
	public int hashCode() {
		int result = getPrefix() != null ? getPrefix().hashCode() : 0;
		result = 31 * result + (getSchema() != null ? getSchema().hashCode() : 0);
		result = 31 * result + (getNamespace() != null ? getNamespace().hashCode() : 0);
		result = 31 * result + (getSourceMetadataReference() != null ? getSourceMetadataReference().hashCode() : 0);
		result = 31 * result + (getBaseQuery() != null ? getBaseQuery().hashCode() : 0);
		result = 31 * result + (getTransformationRuleID() != null ? getTransformationRuleID().hashCode() : 0);
		result = 31 * result + (isEnabled() ? 1 : 0);
		return result;
	}


}
