package eu.dnetlib.enabling.tools.registration;

import java.io.InputStream;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;
import javax.xml.transform.dom.DOMResult;
import javax.xml.ws.Endpoint;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.tools.AbstractBaseService;
import eu.dnetlib.enabling.tools.BaseServiceUtils;
import eu.dnetlib.enabling.tools.HNMLocator;
import eu.dnetlib.enabling.tools.NullHNMLocator;
import eu.dnetlib.rmi.common.BaseService;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.enabling.ISRegistryService;
import eu.dnetlib.rmi.enabling.ISSNService;
import eu.dnetlib.soap.EndpointReferenceBuilder;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

/**
 * This class takes care of registering a service.
 *
 * TODO: merge the implementation
 *
 * @author marko
 *
 */
public class ServiceRegistrator {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(ServiceRegistrator.class);

	/**
	 * locator.
	 */
	private UniqueServiceLocator serviceLocator;

	/**
	 * epr builder.
	 */
	private EndpointReferenceBuilder<Endpoint> eprBuilder;

	/**
	 * component which finds an hnm profile.
	 */
	private HNMLocator hnmLocator = new NullHNMLocator();



	public String registerService(final BaseService service, final Endpoint endpoint) throws Exception {
		if (service instanceof AbstractBaseService) {
			AbstractBaseService abs = (AbstractBaseService) service;
			return registerService(BaseServiceUtils.getServiceName(service.getClass()), this.eprBuilder.getEndpointReference(endpoint),
					abs.getServiceProperties(), abs.getExtraProtocols());
		}
		return registerService(BaseServiceUtils.getServiceName(service.getClass()), this.eprBuilder.getEndpointReference(endpoint), new HashMap<>(),
				new HashMap<>());
	}

	private String registerService(final String serviceName, final W3CEndpointReference epr, final Map<String, String> serviceProperties,
			final Map<String, String> extraProtocols) throws Exception {

		ensureSchemaExists(serviceName);

		final DOMResult result = new DOMResult();
		epr.writeTo(result);

		final InputStream templateStream = getClass().getResourceAsStream("ServiceProfileTemplate.st");
		if (templateStream == null) { throw new IllegalStateException("cannot find service profile template"); }

		final StringWriter buffer = new StringWriter();
		IOUtils.copy(templateStream, buffer);

		final StringTemplate templ = new StringTemplate(buffer.toString());

		final String resourceType = serviceName + "ResourceType";

		final String address = result.getNode().getChildNodes().item(0).getChildNodes().item(0).getTextContent();
		final String hnmId = this.hnmLocator.getHNMForUrl(address);

		// skip registration if there is no HNM yet.
		if (hnmId == null) { return null; }

		templ.setAttribute("resourceType", resourceType);
		templ.setAttribute("serviceName", serviceName);
		templ.setAttribute("address", address);
		templ.setAttribute("protocols", extraProtocols);
		templ.setAttribute("parentId", hnmId);
		templ.setAttribute("properties", serviceProperties);

		log.debug("template: " + templ.toString());

		final String profId = this.serviceLocator.getService(ISRegistryService.class, true).registerProfile(templ.toString());

		final String topic = "UPDATE/" + resourceType + "/" + profId + "/RESOURCE_PROFILE/BODY/BLACKBOARD/LAST_REQUEST";

		getServiceLocator().getService(ISSNService.class, true).subscribe(epr, topic, 0);

		log.info("registered profile for " + serviceName + ": " + profId);

		return profId;
	}

	/**
	 * Check that the service schema for this service already exists, and create it if it doesn't.
	 *
	 * @param serviceName
	 *            service name
	 */
	protected void ensureSchemaExists(final String serviceName) throws Exception {
		try {
			this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
					"//*[local-name() = 'complexType' and @name = 'RESOURCE_TYPEType']//*[local-name() = 'enumeration' and @value = '" + serviceName
							+ "ResourceType']");
			log.info("schema for " + serviceName + " appears to exist");
		} catch (final ISLookUpDocumentNotFoundException e) {
			registerServiceSchema(serviceName);
		}
	}

	private void registerServiceSchema(final String serviceName) throws Exception {

		final InputStream schemaStream = getClass().getResourceAsStream("ServiceProfileSchemaTemplate.st");
		if (schemaStream == null) { throw new IllegalStateException("cannot find service profile schema template"); }

		final StringTemplate schema = new StringTemplate(IOUtils.toString(schemaStream));

		final String resourceType = serviceName + "ResourceType";
		schema.setAttribute("resourceType", resourceType);

		if (this.serviceLocator == null) {
			log.error("************* SERVICE LOCATOR IS NULL:" + serviceName);
			return;
		}
		final ISRegistryService registry = this.serviceLocator.getService(ISRegistryService.class, true);
		if (registry == null) {
			log.error("************* REGISTRY SERVICE IS NULL");
			return;
		}

		registry.addResourceType(resourceType, schema.toString());

		log.info("registered schema for " + serviceName);

	}

	public EndpointReferenceBuilder<Endpoint> getEprBuilder() {
		return this.eprBuilder;
	}

	@Required
	public void setEprBuilder(final EndpointReferenceBuilder<Endpoint> eprBuilder) {
		this.eprBuilder = eprBuilder;
	}

	public HNMLocator getHnmLocator() {
		return this.hnmLocator;
	}

	public void setHnmLocator(final HNMLocator hnmLocator) {
		this.hnmLocator = hnmLocator;
	}

	public UniqueServiceLocator getServiceLocator() {
		return this.serviceLocator;
	}

	@Required
	public void setServiceLocator(final UniqueServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}
}
