package eu.dnetlib.miscutils.functional.xml;

import java.io.*;
import java.util.Map;
import java.util.function.Function;
import javax.xml.transform.*;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;



/**
 * This function applies a stylesheet to something which can be transformed to a javax.xml.Source.
 * 
 * <p>Subclasses are specialized and know how to transform K into a Source</p>
 * 
 * @author marko
 *
 * @param <K>
 */
public abstract class AbstractApplyXslt<K> implements Function<K, String > {
	private static final String UNKNOWN_XSLT_NAME = "unknown xslt name";

	private static final Log log = LogFactory.getLog(AbstractApplyXslt.class); // NOPMD by marko on 11/24/08 5:02 PM

	private Transformer transformer;

	private TransformerFactory transformerFactory;
	
	/**
	 * optional, useful to keep track of xslt name for debugging purposes.
	 */
	private String xsltName;

	public AbstractApplyXslt(final Resource xslt, final TransformerFactory transformerFactory) {
		this(xslt, null, transformerFactory);
	}

	public AbstractApplyXslt(final Resource xslt, final Map<String, String> parameters, final TransformerFactory transformerFactory) {
		this(new StreamSource(getInputStream(xslt)), getFileName((ClassPathResource) xslt), parameters, transformerFactory);
	}
	
	public AbstractApplyXslt(final String xslt, final TransformerFactory transformerFactory) {
		this(xslt, UNKNOWN_XSLT_NAME, transformerFactory);
	}

	public AbstractApplyXslt(final String xslt, final String name, final TransformerFactory transformerFactory) {
		this(xslt, name, null, transformerFactory);
	}

	public AbstractApplyXslt(final String xslt, final String name, final Map<String, String> parameters, final TransformerFactory transformerFactory) {
		this(new StreamSource(new StringReader(xslt)), name, parameters, transformerFactory);
	}

	public AbstractApplyXslt(final Source xslt, final TransformerFactory transformerFactory) {
		this(xslt, UNKNOWN_XSLT_NAME, transformerFactory);
	}

	public AbstractApplyXslt(final Source xslt, final String name, final TransformerFactory transformerFactory) {
		this(xslt, name, null, transformerFactory);
	}

	/**
	 * Base method for all the others.
	 * @param xslt stylesheet as Source
	 * @param name name of the stylesheet
	 * @param parameters map of parameters to pass to the xslt
	 * @param transformerFactory factory to create Transformer instance. Must be a Saxon transformer Factory.
	 */
	public AbstractApplyXslt(final Source xslt, String name, Map<String, String> parameters, TransformerFactory transformerFactory) {
		this.transformerFactory = transformerFactory;
		try {
			this.xsltName = name;
			transformer = getTransformerFactory().newTransformer(xslt);
			if (! UNKNOWN_XSLT_NAME.equals(name))
				transformer.setURIResolver(new ExternalResourceURIResolver(name.replaceFirst("[^/]+$", "")));
			if(parameters != null)
				for(Map.Entry<String, String> parameter : parameters.entrySet())
					transformer.setParameter(parameter.getKey(), parameter.getValue());
			
		} catch (final Throwable e) {
			log.error("Problems with transformer!\n" + xslt + "--" + name, e);
			log.error(xsltDump(xslt));
			throw new IllegalStateException(e);
		}
	}
	
	/**
	 * Used only to convert checked to unchecked exception.
	 *
	 * @param xslt
	 * @return
	 */
	private static InputStream getInputStream(final Resource xslt) {
		try {
			return xslt.getInputStream();
		} catch (final IOException e) {
			throw new IllegalArgumentException(e);
		}
	}

	private static String getFileName(ClassPathResource xslt) {
		return xslt.getPath();
	}

	private String xsltDump(Source xslt) {
		Transformer transformer;
		try {
			transformer = getTransformerFactory().newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			StringWriter outWriter = new StringWriter();
			Result result = new StreamResult(outWriter);
			transformer.transform(xslt, result);
			StringBuffer sb = outWriter.getBuffer();
			return sb.toString();
		} catch (TransformerConfigurationException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (TransformerException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return "error dumping the xslt";
	}

	@Override
	public String apply(final K input) {
		synchronized (transformer) {
			try {
				final StringWriter output = new StringWriter();
				transformer.transform(toStream(input), new StreamResult(output));
				return output.toString();
			} catch (final TransformerException e) {
				log.error("cannot transform record", e);
				log.error(input.toString());
				return "";
			}
		}
	}

	public abstract Source toStream(K input);

	public abstract String toString(K input);
	
	public TransformerFactory getTransformerFactory() {
		return transformerFactory;
	}

	public void setTransformerFactory(final TransformerFactory transformerFactory) {
		this.transformerFactory = transformerFactory;
	}

	public String getXsltName() {
		return xsltName;
	}

	public void setXsltName(String xsltName) {
		this.xsltName = xsltName;
	}

	/**
	 * This class contains the login enabling imports of peer external resources for the current xslt (<xsl:import>)
	 * The method resolve() first looks for the resource within the file system; if this fails, it looks in the classpath.
	 *
	 * @author Andrea Mannocci
	 */
	private class ExternalResourceURIResolver implements URIResolver {

		private final String xsltBasePath;

		public ExternalResourceURIResolver(String xsltBasePath) {
			this.xsltBasePath = xsltBasePath;
		}

		@Override
		public Source resolve(String href, String base) throws TransformerException {
			String externalResource = this.xsltBasePath + href;
			try {
				log.debug("trying to load external resource from file system: " + externalResource);
				return new StreamSource(new File(externalResource));
			} catch (final Throwable e) {
				log.debug("trying to load external resource from file system: " + externalResource);
				return new StreamSource(getInputStream(new ClassPathResource(externalResource)));
			}
		}
	}

}
