package eu.dnetlib.enabling.resultset.client;

import java.io.InputStream;
import java.util.List;
import java.util.stream.Collectors;

import com.google.gson.Gson;

import eu.dnetlib.enabling.resultset.ResultSetResponse;
import eu.dnetlib.rmi.common.ResultSetException;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

public class HttpResultSetClientIterator<T> extends AbstractResultSetClientIterator<T> {

	private static final int PAGE_SIZE = 20;

	private final String id;
	private final String baseUrl;
	private final Class<T> clazz;

	private final CloseableHttpClient client = HttpClients.createDefault();

	public HttpResultSetClientIterator(final String id, final String baseUrl, final Class<T> clazz) {
		super();
		this.id = id;
		this.baseUrl = baseUrl;
		this.clazz = clazz;
	}

	@Override
	protected List<T> nextPage() throws ResultSetException {
		final HttpGet method = new HttpGet(getBaseUrl() + "/" + getId() + "/next/" + PAGE_SIZE);
        method.addHeader("Accept", "application/json");

		try(CloseableHttpResponse httpResponse = client.execute(method)) {

			int statusCode = httpResponse.getStatusLine().getStatusCode();
			if (HttpStatus.SC_OK != statusCode) {
				throw new ResultSetException("Error " + statusCode + " dowloading url: " + getBaseUrl());
			}

			try(InputStream responseBody = httpResponse.getEntity().getContent()) {
				final String json = IOUtils.toString(responseBody);
				final Gson gson = new Gson();
				final ResultSetResponse response = gson.fromJson(json, ResultSetResponse.class);
				return response.getElements()
						.stream()
						.map(s -> gson.fromJson(s, getClazz()))
						.collect(Collectors.toList());
			}
		} catch (Throwable e) {
			throw new ResultSetException("Error dowloading url: " + getBaseUrl(), e);
		}
	}

	public String getId() {
		return this.id;
	}

	public String getBaseUrl() {
		return this.baseUrl;
	}

	public Class<T> getClazz() {
		return this.clazz;
	}

}
