package eu.dnetlib.enabling.resultset.factory;

import java.util.Map;
import java.util.function.Function;
import java.util.stream.StreamSupport;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;

import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.enabling.resultset.listener.IterableResultSetListener;
import eu.dnetlib.enabling.resultset.listener.ResultSetListener;
import eu.dnetlib.enabling.resultset.registry.ResultSetRegistry;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.rmi.common.ResultSet;

public class ResultSetFactory {

	private ResultSetRegistry resultSetRegistry;
	private ResultSetClient resultSetClient;
	private String baseUrl;
	@Autowired
	private TransformerFactory saxonTransformerFactory;

	public <T> ResultSet<T> registerResultSet(final ResultSetListener<T> rs) {
		final String rsId = this.resultSetRegistry.registerResultSet(rs);
		return new ResultSet<T>(rsId, this.baseUrl);
	}

	public <T> ResultSet<T> createResultSet(final ResultSetListener<T> listener) {
		return registerResultSet(listener);
	}

	public <T> ResultSet<T> createResultSet(final Iterable<T> iter) {
		return registerResultSet(new IterableResultSetListener<T>(iter));
	}

	public <T, K> ResultSet<K> map(final ResultSet<?> rsIn, final Class<T> clazzIn, final Function<T, K> mapper) {
		final Iterable<T> iterIn = this.resultSetClient.iter(rsIn, clazzIn);
		return createResultSet(() -> StreamSupport.stream(iterIn.spliterator(), false).map(mapper).iterator());
	}

	public ResultSet<String> xsltMap(final ResultSet<?> rsIn, final String xslt) {
		return map(rsIn, String.class, new ApplyXslt(xslt, saxonTransformerFactory));
	}

	public ResultSet<String> xsltMap(final ResultSet<?> rsIn, final String xslt, final Map<String, String> params) {
		return map(rsIn, String.class, new ApplyXslt(xslt, null, params, saxonTransformerFactory));
	}

	public ResultSet<String> xsltMap(final ResultSet<?> rsIn, final Resource xsltResource) {
		return map(rsIn, String.class, new ApplyXslt(xsltResource,saxonTransformerFactory));
	}

	public ResultSet<String> xsltMap(final ResultSet<?> rsIn, final Resource xsltResource, final Map<String, String> params) {
		return map(rsIn, String.class, new ApplyXslt(xsltResource, params, saxonTransformerFactory));
	}

	public ResultSet<String> xsltMap(final ResultSet<?> rsIn, final DOMSource domSource) {
		return map(rsIn, String.class, new ApplyXslt(domSource,saxonTransformerFactory));
	}

	public ResultSetRegistry getResultSetRegistry() {
		return this.resultSetRegistry;
	}

	@Required
	public void setResultSetRegistry(final ResultSetRegistry resultSetRegistry) {
		this.resultSetRegistry = resultSetRegistry;
	}

	public String getBaseUrl() {
		return this.baseUrl;
	}

	@Required
	public void setBaseUrl(final String baseUrl) {
		this.baseUrl = baseUrl;
	}

	public ResultSetClient getResultSetClient() {
		return this.resultSetClient;
	}

	@Required
	public void setResultSetClient(final ResultSetClient resultSetClient) {
		this.resultSetClient = resultSetClient;
	}

}
