package eu.dnetlib.enabling.resultset.client;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Required;

import com.google.gson.Gson;

import eu.dnetlib.enabling.resultset.ResultSetInfo;
import eu.dnetlib.enabling.resultset.listener.ResultSetListener;
import eu.dnetlib.enabling.resultset.registry.ResultSetRegistry;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.common.ResultSetException;

public class ResultSetClient {

	private ResultSetRegistry resultSetRegistry;

	private static final Log log = LogFactory.getLog(ResultSetClient.class);

	public <T> Iterable<T> iter(final ResultSet<?> resultSet, final Class<T> clazz) {
		try {
			if (this.resultSetRegistry.contains(resultSet.getId())) {
				final ResultSetListener<?> listener = this.resultSetRegistry.getResultSetById(resultSet.getId());
				return () -> new LocalResultSetClientIterator<>(listener, clazz);
			} else {
				return () -> new HttpResultSetClientIterator<>(resultSet.getId(), resultSet.getBaseUrl(), clazz);
			}
		} catch (final Throwable e) {
			log.error("ITER: Error accessing resultset: " + resultSet.getId());
			throw new RuntimeException(e);
		}
	}

	public ResultSetInfo info(final ResultSet<?> resultSet) {
		try {
			if (this.resultSetRegistry.contains(resultSet.getId())) {
				final ResultSetListener<?> listener = this.resultSetRegistry.getResultSetById(resultSet.getId());
				return new ResultSetInfo(resultSet.getId(), listener);
			} else {
				return invokeRemoteInfo(resultSet);
			}
		} catch (final Throwable e) {
			log.error("INFO: Error accessing resultset: " + resultSet.getId());
			throw new RuntimeException(e);
		}
	}

	private ResultSetInfo invokeRemoteInfo(final ResultSet<?> rs) throws ResultSetException {
		final HttpGet method = new HttpGet(rs.getBaseUrl() + "/" + rs.getId() + "/info");
		try(CloseableHttpClient httpClient = HttpClients.createDefault()) {
			try(CloseableHttpResponse response = httpClient.execute(method)) {

				int statusCode = response.getStatusLine().getStatusCode();

				if (HttpStatus.SC_OK != statusCode) {
					throw new ResultSetException("Error " + statusCode + " dowloading url: " + rs.getBaseUrl());
				}
				try(InputStream responseBody = response.getEntity().getContent()) {
					String json = IOUtils.toString(responseBody);
					return new Gson().fromJson(json, ResultSetInfo.class);
				}
			} catch (final Throwable e) {
				throw new ResultSetException("Error dowloading url: " + rs.getBaseUrl(), e);
			}
		} catch (IOException e) {
			log.error("Can't close connections gracefully", e);
			throw new ResultSetException("Can't close connections gracefully", e);
		}
	}

	public ResultSetRegistry getResultSetRegistry() {
		return this.resultSetRegistry;
	}

	@Required
	public void setResultSetRegistry(final ResultSetRegistry resultSetRegistry) {
		this.resultSetRegistry = resultSetRegistry;
	}

}
