package eu.dnetlib.enabling.resultset.registry;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import eu.dnetlib.enabling.resultset.listener.ResultSetListener;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.rmi.common.ResultSetException;

public class ResultSetRegistryImpl implements ResultSetRegistry {

	private static final String RS_PREFIX = "rs-";

	private long ttl = 600000;

	private static final Log log = LogFactory.getLog(ResultSetRegistry.class);

	private class RsCacheElem {

		private final ResultSetListener<?> listener;
		private long lastDate = DateUtils.now();

		public RsCacheElem(final ResultSetListener<?> listener) {
			this.listener = listener;
		}

		public ResultSetListener<?> getListener() {
			this.lastDate = DateUtils.now();
			return this.listener;
		}

		public boolean isOld(final long now) {
			return (now - this.lastDate) > ResultSetRegistryImpl.this.ttl;
		}
	}

	private final Map<String, RsCacheElem> cache = new HashMap<>();

	@Override
	public String registerResultSet(final ResultSetListener<?> listener) {
		verifyCache();
		final String newId = RS_PREFIX + UUID.randomUUID();
		this.cache.put(newId, new RsCacheElem(listener));

		log.debug("New resultset created: " + newId);

		return newId;
	}

	private void verifyCache() {
		final long now = DateUtils.now();
		final Set<String> toDelete = this.cache.entrySet()
				.stream()
				.filter(e -> e.getValue().isOld(now))
				.map(e -> e.getKey())
				.collect(Collectors.toSet());

		if (!toDelete.isEmpty()) {
			log.debug("The following resultsets are expired: " + toDelete);
		}

		toDelete.forEach(this.cache::remove);
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> ResultSetListener<T> getResultSetById(final String rsId) throws ResultSetException {
		final RsCacheElem elem = this.cache.get(rsId);
		if (elem != null) {
			final ResultSetListener<T> rs = (ResultSetListener<T>) elem.getListener();
			if (rs != null) { return rs; }
		}
		throw new ResultSetException("Missing resultset with id " + rsId);
	}

	@Override
	public boolean contains(final String rsId) {
		return this.cache.get(rsId) != null;
	}

	public long getTtl() {
		return this.ttl;
	}

	@Required
	public void setTtl(final long ttl) {
		this.ttl = ttl;
	}

}
