package eu.dnetlib.data.mdstore.modular.mongodb;

import java.util.Arrays;
import java.util.List;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.QueryBuilder;
import com.mongodb.client.*;
import eu.dnetlib.data.mdstore.modular.RecordParser;
import eu.dnetlib.data.mdstore.modular.connector.MDStore;
import eu.dnetlib.enabling.resultset.listener.ResultSetListener;
import eu.dnetlib.enabling.tools.DnetStreamSupport;
import eu.dnetlib.rmi.data.DocumentNotFoundException;
import eu.dnetlib.rmi.data.MDStoreServiceException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

public class MongoMDStore implements MDStore {

	private static final int BULK_SIZE = 100;
	private static final Log log = LogFactory.getLog(MongoMDStore.class);
	private static List<String> requiredIndices = Arrays.asList("{ \"id\" : 1}", "{ \"timestamp\" : 1}", "{ \"originalId\" : 1}");
	private final boolean discardRecords;
	private String id;
	private MongoDatabase mongoDatabase;
	private MongoCollection<DBObject> collection;
	private MongoCollection<DBObject> discardedCollection;

	private RecordParser recordParser;

	public MongoMDStore(final String id,
			final MongoCollection<DBObject> collection,
			final RecordParser recordParser,
			final boolean discardRecords,
			final MongoDatabase mongoDatabase) {
		this.id = id;
		this.mongoDatabase = mongoDatabase;
		this.collection = collection;
		this.discardedCollection = this.mongoDatabase.getCollection("discarded-" + StringUtils.substringBefore(id, "_"), DBObject.class);
		this.recordParser = recordParser;
		this.discardRecords = discardRecords;
	}

	@Override
	public void feed(final Iterable<String> records, final boolean incremental) {
		// TODO: remove incremental from MDStore API. It is used in MDStoreModular. Useless here.

		ensureIndices();

		final BlockingQueue<Object> queue = new ArrayBlockingQueue<>(100);
		final Object sentinel = new Object();
		final Thread background = new Thread(() -> {
			final MongoBulkWritesManager bulkWritesManager =
					new MongoBulkWritesManager(collection, discardedCollection, BULK_SIZE, recordParser, discardRecords);
			int count = 0;
			while (true) {
				try {
					final Object record = queue.take();
					if (record == sentinel) {
						bulkWritesManager.flushBulks(mongoDatabase);
						break;
					}
					count++;
					bulkWritesManager.insert((String) record);
				} catch (final InterruptedException e) {
					log.fatal("got exception in background thread", e);
					throw new IllegalStateException(e);
				}
			}
			log.debug(String.format("extracted %s records from feeder queue", count));
		});

		background.start();
		try {
			log.info("feeding mdstore " + id);
			if (records != null) {
				for (final String record : records) {
					queue.put(record);
				}
			}
			queue.put(sentinel);
			log.info("finished feeding mdstore " + id);

			background.join();
		} catch (final InterruptedException e) {
			throw new IllegalStateException(e);
		}
		// double check
		ensureIndices();
		collection.createIndex(new BasicDBObject("id", 1));
	}

	public void ensureIndices() {
		for (final String key : Arrays.asList("id", "timestamp", "originalId")) {
			collection.createIndex(new BasicDBObject(key, 1));
		}
	}

	public boolean isIndexed() {
		final ListIndexesIterable<DBObject> listIndexesIterable = collection.listIndexes(DBObject.class);

		Stream<DBObject> inputStream = StreamSupport.stream(Spliterators.spliteratorUnknownSize(listIndexesIterable.iterator(), Spliterator.ORDERED), false);
		final List<String> key = inputStream.map(dbo -> new BasicDBObject(dbo.toMap()).getString("key")).collect(Collectors.toList());
		return requiredIndices.stream().allMatch(key::contains);
	}

	/**
	 * Method searches for the given string grep into this collection and replaces it with the given replacement.
	 *
	 * @param grep    the string to search
	 * @param replace the replacement
	 */
	public void replace(final String grep, final String replace) {
		final Pattern regex = Pattern.compile(grep, Pattern.MULTILINE);
		BasicDBObject query = (BasicDBObject) QueryBuilder.start("body").regex(regex).get();
		final FindIterable<DBObject> matches = collection.find(query, DBObject.class);
		if (log.isDebugEnabled())
			log.debug("FOUND: " + collection.count(query));
		for (final DBObject match : matches) {
			final DBObject o = new BasicDBObject(match.toMap());
			o.put("body", regex.matcher((String) match.get("body")).replaceAll(replace));
			collection.findOneAndReplace(new BasicDBObject("_id", o.get("_id")), o);
		}
	}

	@Override
	public ResultSetListener<String> deliver(final String from, final String until, final String recordFilter) {
		return deliver(from, until, recordFilter, new SerializeMongoRecord());
	}

	@Override
	public ResultSetListener<String> deliverIds(final String from, final String until, final String recordFilter) {
		return deliver(from, until, recordFilter, new SerializeMongoRecordId());
	}

	public ResultSetListener deliver(final String from, final String until, final String recordFilter, final Function<DBObject, String> serializer) {
		try {
			final Pattern filter = (recordFilter != null) && (recordFilter.length() > 0) ? Pattern.compile(recordFilter, Pattern.MULTILINE) : null;

			return new MongoResultSetListener(collection, parseLong(from), parseLong(until), filter, serializer, BULK_SIZE);
		} catch(Throwable e) {
			throw new RuntimeException(e);
		}
	}

	private Long parseLong(final String s) throws MDStoreServiceException {
		if (StringUtils.isBlank(s)) {
			return null;
		}
		try {
			return Long.valueOf(s);
		} catch (NumberFormatException e) {
			throw new MDStoreServiceException("Invalid date, expected java.lang.Long, or null", e);
		}
	}

	@Override
	public Iterable<String> iterate() {
		return () -> {
            final MongoCursor<DBObject> i = collection.find().noCursorTimeout(true).batchSize(BULK_SIZE).iterator();
            return DnetStreamSupport.generateStreamFromIterator(i)
					.map(it -> (String) it.get("body"))
					.iterator();
		};
	}

	@Override
	public void deleteRecord(final String recordId) {
		collection.deleteOne(new BasicDBObject("id", recordId));
	}

	@Override
	public String getRecord(final String recordId) throws DocumentNotFoundException {
		final DBObject obj = collection.find(new BasicDBObject("id", recordId)).first();
		if (obj == null || !obj.containsField("body")) throw new DocumentNotFoundException(String.format(
				"The document with id '%s' does not exist in mdstore: '%s'", recordId, id));
		final String body = (String) obj.get("body");
		if (body.trim().length() == 0) throw new DocumentNotFoundException(String.format("The document with id '%s' does not exist in mdstore: '%s'",
				recordId, id));
		return new SerializeMongoRecord().apply(obj);
	}

	@Override
	public void truncate() {
		collection.drop();
		discardedCollection.drop();
	}

	public DBObject getMDStoreMetadata() {
		return mongoDatabase.getCollection("metadata", DBObject.class).find(new BasicDBObject("mdId", getId())).first();
	}

	@Override
	public String getFormat() {
		return (String) getMDStoreMetadata().get("format");
	}

	@Override
	public String getInterpretation() {
		return (String) getMDStoreMetadata().get("interpretation");
	}

	@Override
	public String getLayout() {
		return (String) getMDStoreMetadata().get("layout");
	}

	@Override
	public String getId() {
		return id;
	}

	public void setId(final String id) {
		this.id = id;
	}

	public MongoCollection<DBObject> getCollection() {
		return collection;
	}

	public void setCollection(final MongoCollection<DBObject> collection) {
		this.collection = collection;
	}

	public RecordParser getRecordParser() {
		return recordParser;
	}

	@Required
	public void setRecordParser(final RecordParser recordParser) {
		this.recordParser = recordParser;
	}

	@Override
	public int getSize() {
		return (int) collection.count();
	}

	public MongoCollection<DBObject> getDiscardedCollection() {
		return discardedCollection;
	}

	public void setDiscardedCollection(final MongoCollection<DBObject> discardedCollection) {
		this.discardedCollection = discardedCollection;
	}

	private class SerializeMongoRecord implements Function<DBObject, String> {

		@Override
		public String apply(final DBObject arg) {
			return (String) arg.get("body");
		}
	}

	private class SerializeMongoRecordId implements Function<DBObject, String> {

		@Override
		public String apply(final DBObject arg) {
			return (String) arg.get("id");
		}
	}

}
