package eu.dnetlib.data.mdstore.modular;

import java.util.Map;

import com.google.common.collect.Maps;
import eu.dnetlib.data.mdstore.modular.action.DoneCallback;
import eu.dnetlib.data.mdstore.modular.action.FailedCallback;
import eu.dnetlib.data.mdstore.modular.connector.MDStore;
import eu.dnetlib.data.mdstore.modular.connector.MDStoreDao;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClient;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.data.MDStoreServiceException;
import eu.dnetlib.rmi.enabling.ISRegistryService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

public class MDStoreFeeder {

	private static final Log log = LogFactory.getLog(MDStoreFeeder.class);

	private MDStoreDao dao;

	@Autowired
	private ResultSetClient resultSetClient;

	private UniqueServiceLocator serviceLocator;

	public void feed(final String mdId,
			final ResultSet<?> resultSet,
			final String storingType,
			final DoneCallback doneCallback,
			final FailedCallback failCallback) throws MDStoreServiceException {
		log.info("Start feeding mdstore " + mdId);
		log.debug("Start feeding mdstore " + mdId + " with resultSet " + resultSet.toJson());

		String transactionId = null;

		try {
			final boolean refresh = "REFRESH".equals(storingType);

			final MDStore mdstore = dao.startTransaction(mdId, refresh);
			transactionId = mdstore.getId();

			final Iterable<String> records = resultSetClient.iter(resultSet, String.class);

			if (refresh) {
				mdstore.truncate();
			}

			mdstore.feed(records, refresh);

			dao.commit(mdstore.getId(), mdId);

			int size = dao.refreshSize(mdId);

			touch(mdId, size);

			log.info("Finished feeding mdstore " + mdId + " - new size: " + size);

			doneCallback.call(buildParams(size));
		} catch (Throwable e) {
			if (transactionId != null) {
				dao.invalidTransaction(transactionId, mdId);
			}
			log.error("Error feeding mdstore: " + mdId);
			failCallback.call(e);
		}
	}

	private Map<String, String> buildParams(final int size) {
		Map<String, String> params = Maps.newHashMap();
		params.put("size", String.valueOf(size));
		return params;
	}

	/**
	 * Sets the last modified date in the profile.
	 *
	 * @param mdId
	 */
	public void touch(final String mdId, final int size) {
		try {
			final String now = DateUtils.now_ISO8601();

			final String mdstoreXUpdate = "for $x in //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + mdId + "']"
					+ "return update value $x//LAST_STORAGE_DATE with '" + now + "'";

			serviceLocator.getService(ISRegistryService.class).executeXUpdate(mdstoreXUpdate);

			touchSize(mdId, size);
		} catch (final Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public void touchSize(final String mdId, final int size) {
		try {
			final String mdstoreNumberXUpdate = "for $x in //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + mdId + "']"
					+ "return update value $x//NUMBER_OF_RECORDS with '" + size + "'";

			serviceLocator.getService(ISRegistryService.class).executeXUpdate(mdstoreNumberXUpdate);
		} catch (final Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public MDStoreDao getDao() {
		return dao;
	}

	@Required
	public void setDao(final MDStoreDao dao) {
		this.dao = dao;
	}

	public UniqueServiceLocator getServiceLocator() {
		return serviceLocator;
	}

	@Required
	public void setServiceLocator(final UniqueServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}

}
