package eu.dnetlib.data.cleaner;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Joiner;
import eu.dnetlib.rmi.data.CleanerException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author michele
 *
 *         Vocabulary rules must be declared in a CleanerDS profile, for each vocabulary must be present the relative VocabularyDS profile:
 *
 *         <RULE xpath="..." vocabularies="VOC1" /> <RULE xpath="..." vocabularies="VOC1, VOC2, VOC3" />
 */

public class VocabularyRule extends XPATHCleaningRule {

	private static final Log log = LogFactory.getLog(VocabularyRule.class); // NOPMD by marko on 11/24/08 5:02 PM
	private final Set<String> vocabularies;
	private final Map<String, String> synonyms = new HashMap<>();
	private final Set<String> validTerms = new HashSet<>();

	public VocabularyRule(final Set<String> vocabularies, final ISLookUpService lookup) throws CleanerException {
		this.vocabularies = vocabularies;
		loadSynonymsAndTerms(lookup);
	}

	@Override
	protected String calculateNewValue(final String oldValue) throws CleanerException {
		log.debug("calculating new value for: " + oldValue);

		if (this.synonyms.isEmpty()) {
			log.debug("Vocabulary terms is void, vocabularies: " + this.vocabularies);
		}

		String newValue = null;

		if (this.synonyms.containsKey(oldValue.toLowerCase())) {
			newValue = this.synonyms.get(oldValue.toLowerCase());
		}

		if (newValue == null) {
			log.debug("Synonym " + oldValue + " not found in vocabulary");
			return oldValue;
		}

		return newValue;
	}

	private void loadSynonymsAndTerms(final ISLookUpService lookup) throws CleanerException {

		for (final String vocabulary : this.vocabularies) {
			try {
				final String query = "for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType')"
						+ "//RESOURCE_PROFILE[.//VOCABULARY_NAME/@code='" + vocabulary + "']//TERM return "
						+ "( concat($x/@code,'|-:-|', $x/@code), concat($x/@english_name,'|-:-|', $x/@code), concat($x/@native_name,'|-:-|', $x/@code), "
						+ "for $y in $x//SYNONYM return concat($y/@term,'|-:-|', $x/@code) )";

				for (final String s : lookup.quickSearchProfile(query)) {
					log.debug("SYNONYM : " + s);
					final String[] arr = s.split("\\|-:-\\|");
					if (arr[0] == null || arr[0].isEmpty()) {
						continue;
					}
					this.synonyms.put(arr[0].toLowerCase(), arr[1]);
					this.validTerms.add(arr[1].toLowerCase());
				}

				log.debug("VOCABULARY " + vocabulary.trim() + " - terms size " + this.synonyms.size());
			} catch (final Exception e) {
				throw new CleanerException("Error obtaining vocabulary " + vocabulary, e);
			}
		}

	}

	@Override
	protected Map<String, String> verifyValue(final String value) throws CleanerException {
		if (this.synonyms.isEmpty()) {
			log.debug("Vocabulary terms is void, vocabularies: " + this.vocabularies);
		}

		if (this.validTerms.contains(value.toLowerCase())) { return null; }

		final Map<String, String> error = new HashMap<String, String>();
		error.put("term", value);
		error.put("vocabularies", this.vocabularies.toString().replaceAll("\\[", "").replaceAll("\\]", ""));
		error.put("xpath", this.getXpath());
		return error;
	}

	public Map<String, String> getVocabularyTerms() {
		return this.synonyms;
	}

	@Override
	public String toString() {
		return "VOCABULARIES: [" + Joiner.on(", ").join(this.vocabularies) + "]";
	}

}
