package eu.dnetlib.dli.collector.plugin;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import eu.dnetlib.dli.DLIUtils;
import eu.dnetlib.dli.resolver.model.CompletionStatus;
import eu.dnetlib.dli.resolver.model.DLIObjectProvenance;
import eu.dnetlib.dli.resolver.model.DLIObjectRelation;
import eu.dnetlib.dli.resolver.model.DLIResolvedObject;
import eu.dnetlib.pid.resolver.model.ObjectType;
import eu.dnetlib.pid.resolver.model.PID;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Arrays;
import java.util.Collections;

public class ScholixJSONParser {
    private static final Log log = LogFactory.getLog(ScholixJSONParser.class);

    public static DLIResolvedObject parse(final String input) {
        final JsonParser parser = new JsonParser();
        final JsonElement parse = parser.parse(input);
        if (!parse.isJsonObject()) {
            return null;
        }
        final JsonObject root = parse.getAsJsonObject();
        final DLIResolvedObject result = new DLIResolvedObject();
        final JsonObject relationType = getJsonObject(root, "RelationshipType");
        if (relationType == null) {
            log.error("Error RelationshipType is null");
            return null;
        }
        final String relationSemantic = getStringFromObject(relationType, "Name");

        final JsonObject source = getJsonObject(root,"Source");
        final JsonObject target = getJsonObject(root,"Target");
        if(source == null || target ==null  ) {
            log.error("Error source or target cannot be missing ");
            return null;
        }
        final ObjectType sourceType = getType(source);
        final PID sourcePID = getPid(source);
        if (sourcePID == null) {
            log.error("Error SOURCE PID cannot be null");
            return null;
        }



        final PID targetPID = getPid(target);
        if (targetPID == null) {
            log.error("Error Target  PID cannot be null");
            return null;
        }
        result.setPid(sourcePID.getId());
        result.setPidType(sourcePID.getType());

        final String title = getStringFromObject(source, "Title");
        final String date = getStringFromObject(source, "PublicationDate");
        result.setTitles(Collections.singletonList(title));
        result.setDate(date);
        result.setType(sourceType);
        final DLIObjectRelation  relation = new DLIObjectRelation();
        relation.setCompletionStatus(CompletionStatus.incomplete.toString());
        relation.setTargetPID(targetPID);
        relation.setTargetType(getType(target));
        relation.setRelationSemantics(relationSemantic);
        try {
            relation.setInverseRelation(DLIUtils.getInverse(relationSemantic));
        } catch (Exception e) {
            log.error("Error on setting inverse", e);
        }
        result.setRelations(Collections.singletonList(relation));
        final DLIObjectProvenance sourceProvenance = new DLIObjectProvenance();
        final JsonObject publisherOBJ = getJsonObject(source, "Publisher");
        if (publisherOBJ!= null)
            sourceProvenance.setPublisher(getStringFromObject(publisherOBJ,"Name"));
        sourceProvenance.setCompletionStatus(CompletionStatus.complete.toString());
        sourceProvenance.setDatasource("Elsevier");
        sourceProvenance.setDatasourceId("dli_________::elsevier");
        result.setDatasourceProvenance(Collections.singletonList(sourceProvenance));
        return result;
    }


    public static PID getPid(final JsonObject root) {
        final  JsonObject identifier = getJsonObject(root,"Identifier");
        if (identifier == null)
            return null;
        final String pid = getStringFromObject(identifier, "ID");
        final String pidType = getStringFromObject(identifier, "IDScheme");
        final String resolvedURL= getStringFromObject(identifier, "IDURL");

        if(pid == null || pidType ==null || StringUtils.isEmpty(pid) ||StringUtils.isEmpty(pidType)) {
            return null;
        }
        return DLIUtils.createCorrectPID(pid, pidType, resolvedURL);
    }

    public static ObjectType getType(final JsonObject root) {
        ObjectType sourceType = ObjectType.unknown;
        final JsonObject sType = getJsonObject(root, "Type");
        if (sType != null) {
            final String sourceTypeString  =   getStringFromObject(sType,"Name");
            if (sourceTypeString!= null)
                sourceType = sourceTypeString.equals("literature")?ObjectType.publication: ObjectType.valueOf(sourceTypeString);
        }
        return sourceType;
    }

    public static String getStringFromObject(final JsonObject root, final String field) {
        if (root.has(field) && ! root.get(field).isJsonNull()) {
            return root.get(field).getAsString();
        }
        return null;
    }

    public static JsonObject getJsonObject(final JsonObject root, final String field) {
        if (root.has(field) && root.get(field).isJsonObject()) {
            return root.get(field).getAsJsonObject();
        }
        return null;
    }




}
