package eu.dnetlib.resolver.mdstore.plugin;

import com.mongodb.DBObject;
import com.mongodb.client.MongoCollection;
import eu.dnetlib.dli.DLIUtils;
import eu.dnetlib.dli.resolver.model.CompletionStatus;
import eu.dnetlib.dli.resolver.model.DLIObjectRelation;
import eu.dnetlib.dli.resolver.model.DLIResolvedObject;
import eu.dnetlib.dli.resolver.model.serializer.DLIResolverSerializer;
import eu.dnetlib.pid.resolver.PIDResolver;
import eu.dnetlib.pid.resolver.mdstore.plugin.AbstractRecordResolver;
import eu.dnetlib.pid.resolver.mdstore.plugin.RecordResolver;
import eu.dnetlib.pid.resolver.mdstore.plugin.ResolverMDStorePlugin;
import eu.dnetlib.pid.resolver.mdstore.plugin.ResolverSerializer;
import eu.dnetlib.pid.resolver.model.ObjectRelation;
import eu.dnetlib.pid.resolver.model.ObjectType;
import eu.dnetlib.pid.resolver.model.PID;
import eu.dnetlib.pid.resolver.model.ResolvedObject;
import eu.dnetlib.resolver.parser.DLIParser;
import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.function.Function;


/**
 * Created by sandro on 9/22/16.
 */
public class DLIRecordResolver extends AbstractRecordResolver {

    private static final Log log = LogFactory.getLog(DLIRecordResolver.class);

	private final DLIParser parser = new DLIParser();

    public DLIRecordResolver(long ts, boolean forceResolver) {
        super(ts);
        this.forceResolver = forceResolver;
    }


    public String resolve(final String inputRecord) {

		final DLIResolvedObject inputObject = (DLIResolvedObject) parser.parse(inputRecord);
		if (inputObject==null)
			return null;
		if (!forceResolver && StringUtils.isNoneBlank(inputObject.getResolvedDate()))
			return null;

		if (!StringUtils.isBlank(inputObject.getPid())) {
			log.debug("trying to resolve " + inputObject.getPid());
		}

		boolean shouldUpdate = false;
		if (inputObject.getCompletionStatus() == null || !inputObject.getCompletionStatus().equals(CompletionStatus.complete.toString())) {
			shouldUpdate = tryToResolveRecord(inputObject);
		}

        return resolveRelations(inputRecord, inputObject, shouldUpdate,s -> {

			try {
				return DLIUtils.getInverse(s);
			} catch (Exception e) {
				log.error("Error on getting Inverse relation from "+s, e);
				return "";
			}
		});

	}

    @Override
    protected Map<String, ObjectType> tryToResolveRelation(final PID currentPid) {
        if (currentPid.getId() != null && currentPid.getId().contains("dli_resolver")) {
			return null;
		}
		log.debug("tryToResolveRelation " + currentPid);

		DLIResolvedObject resolvedObject = (DLIResolvedObject) pluginResolver.get(0).retrievePIDfromCache(currentPid.getId(), currentPid.getType());
		final Map<String, ObjectType> result_from_cache = fixResolvedRelation(currentPid, resolvedObject);
		if (result_from_cache != null) return result_from_cache;
		for (PIDResolver resolver : pluginResolver) {
            final DLIResolvedObject currentIdentifier = (DLIResolvedObject) resolver.retrievePID(currentPid.getId(), currentPid.getType(), offline);
			final Map<String, ObjectType> result = fixResolvedRelation(currentPid, currentIdentifier);
			if (result != null) return result;
		}
		return null;
	}

	private Map<String, ObjectType> fixResolvedRelation(PID currentPid, DLIResolvedObject currentIdentifier) {
		if (currentIdentifier != null && !StringUtils.isBlank(currentIdentifier.getPid()) && currentIdentifier.getPid().toLowerCase().equals(currentPid.getId().toLowerCase())) {
			final HashMap<String, ObjectType> result = new HashMap<>();
			result.put("dli_resolver::" + DLIUtils.generateIdentifier(currentIdentifier.getPid(), currentIdentifier.getPidType()),
					currentIdentifier.getType());
			return result;
		}
		return null;
	}

	private boolean tryToResolveRecord(final DLIResolvedObject object) {

		DLIResolvedObject resolvedObject1 = (DLIResolvedObject) pluginResolver.get(0).retrievePIDfromCache(object.getPid(), object.getPidType());
		if (fixResolvedObject(object, pluginResolver.get(0), resolvedObject1)) return true;

		for (PIDResolver resolver : pluginResolver) {
            final DLIResolvedObject resolvedObject = (DLIResolvedObject) resolver.retrievePID(object.getPid(), object.getPidType(), offline);
			if (fixResolvedObject(object, resolver, resolvedObject)) return true;
		}
		log.debug("Record NOT Resolved  PID: " + object.getPid());
		return false;
	}

	private boolean fixResolvedObject(DLIResolvedObject object, PIDResolver resolver, DLIResolvedObject resolvedObject) {
		if (resolvedObject != null &&
				resolvedObject.getCompletionStatus() != null &&
				resolvedObject.getCompletionStatus().equals(CompletionStatus.complete.toString())) {
			{
				object.setAuthors(resolvedObject.getAuthors());
				object.setTitles(resolvedObject.getTitles());
				object.setCompletionStatus(resolvedObject.getCompletionStatus());
				object.setDate(resolvedObject.getDate());
				object.getDatasourceProvenance().addAll(resolvedObject.getDatasourceProvenance());
				object.setDescription(resolvedObject.getDescription());
				object.setSubjects(resolvedObject.getSubjects());
				object.setType(resolvedObject.getType());
				log.debug("Record Resolved by " + resolver.getClass().getCanonicalName() + "  PID: " + object.getPid());
				return true;
			}
		}
		return false;
	}

	public void setPluginResolver(final List<PIDResolver> pluginResolver) {
		this.pluginResolver = pluginResolver;
	}


	public BlockingQueue<DBObject> getInputQueue() {
		return inputQueue;
	}

	public void setInputQueue(final BlockingQueue<DBObject> inputQueue) {
		this.inputQueue = inputQueue;
	}

	public MongoCollection<DBObject> getOutputCollection() {
		return outputCollection;
	}

	public void setOutputCollection(final MongoCollection<DBObject> outputCollection) {
		this.outputCollection = outputCollection;
	}

	public void setSerializer(final ResolverSerializer serializer) {
        final DLIResolverSerializer tmp = new DLIResolverSerializer();
        tmp.setPmfTemplate(new StringTemplate(((DLIResolverSerializer) serializer).getPmfTemplate().getTemplate()));
        tmp.setDmfTemplate(new StringTemplate(((DLIResolverSerializer) serializer).getDmfTemplate().getTemplate()));
        tmp.setScholixTemplate(new StringTemplate(((DLIResolverSerializer) serializer).getScholixTemplate().getTemplate()));
        this.serializer = tmp;
	}
}
