package eu.dnetlib.data.transform;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message.Builder;
import eu.dnetlib.data.graph.model.DNGFUtils;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.DNGFProtos.DNGFEntity;
import eu.dnetlib.data.proto.DatasetProtos.Dataset;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import eu.dnetlib.data.proto.FieldTypeProtos.StringField;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.PersonProtos.Person.CoAuthor;
import eu.dnetlib.data.proto.PersonProtos.Person.MergedPerson;
import eu.dnetlib.data.proto.PersonProtos.Person.Metadata;
import eu.dnetlib.data.proto.PublicationProtos.Publication;
import eu.dnetlib.data.proto.SpecialTrustProtos.SpecialTrust;
import eu.dnetlib.pace.config.DedupConfig;
import org.apache.commons.lang3.StringUtils;

public class DNGFEntityMerger {

	private static final String DEDUP_CLASSID = "sysimport:dedup";

	private static final String DNET_PROVENANCE_SCHEME = "dnet:provenanceActions";

	private final Predicate<StringField> skipEmptyStringField = s -> (s != null) && (s.getValue() != null) && !s.getValue().isEmpty();

	private final Predicate<String> skipEmptyString = s -> StringUtils.isNotBlank(s);

	public static DNGF.Builder merge(final String id, final Iterable<DNGF> entities) {
		return merge(null, id, entities);
	}

	public static DNGF.Builder merge(final DedupConfig dedupConf, final String id, final Iterable<DNGF> entities) {
		return new DNGFEntityMerger().mergeEntities(dedupConf, id, entities);
	}

	public static DNGF.Builder merge(final DNGF.Builder builder) {
		return new DNGFEntityMerger().doMergeEntities(builder);
	}

	public DNGF.Builder mergeEntities(final DedupConfig dedupConf, final String id, final Iterable<DNGF> entities) {

		DNGF.Builder builder = DNGF.newBuilder();
		String trust = "0.0";

        //TODO This set has been changed for DLI, but is wrong
        //for openaire before was: for (final DNGF dngf : TrustOrdering.sort(entities))
        for (final DNGF dngf : entities) {
            // doublecheck we're dealing only with main entities
			if (!dngf.getKind().equals(Kind.entity)) throw new IllegalArgumentException("expected DNGFEntity!");

			final String currentTrust = dngf.getDataInfo().getTrust();

			if (!currentTrust.equals(SpecialTrust.NEUTRAL.toString())) {
				trust = currentTrust;
			}
			builder.mergeFrom(dngf);
			if (StringUtils.isBlank(currentTrust))
				builder.getDataInfoBuilder().setTrust("0.8");
		}

		builder = doMergeEntities(builder);
		builder.getEntityBuilder().setId(id);
		builder.getDataInfoBuilder()
				.setInferred(true)
				.setDeletedbyinference(false)
				.setTrust(trust)
				.setInferenceprovenance(dedupConf != null ? dedupConf.getWf().getConfigurationId() : "")
				.setProvenanceaction(getProvenanceAction());

		if ((dedupConf != null) && dedupConf.getWf().isIncludeChildren()) {
			for (final DNGF dngf : Iterables.limit(entities, dedupConf.getWf().getMaxChildren())) {
				builder.getEntityBuilder().addChildren(dngf.getEntity());
			}
		}

		return builder;
	}

	private Qualifier.Builder getProvenanceAction() {
		return Qualifier.newBuilder().setClassid(DEDUP_CLASSID).setClassname(DEDUP_CLASSID).setSchemeid(DNET_PROVENANCE_SCHEME)
				.setSchemename(DNET_PROVENANCE_SCHEME);
	}

	public DNGF.Builder doMergeEntities(final DNGF.Builder builder) {

		for (final String field : DNGFUtils.getFieldNames(DNGFEntity.getDescriptor(), DNGFEntity.COLLECTEDFROM_FIELD_NUMBER)) {
			setKeyValues(builder.getEntityBuilder(), field);
		}
		for (final String field : DNGFUtils.getFieldNames(DNGFEntity.getDescriptor(), DNGFEntity.PID_FIELD_NUMBER)) {
			setStructuredProperty(builder.getEntityBuilder(), field);
		}
		for (final String field : DNGFUtils.getFieldNames(DNGFEntity.getDescriptor(), DNGFEntity.ORIGINALID_FIELD_NUMBER)) {
			setUniqueString(builder.getEntityBuilder(), field);
		}

		switch (builder.getEntity().getType()) {
		case datasource:
			break;
		case organization:
			break;
		case person:
			final Person.Builder person = builder.getEntityBuilder().getPersonBuilder().setAnchor(true);

			for (final String field : DNGFUtils.getFieldNames(Person.Metadata.getDescriptor(), Metadata.SECONDNAMES_FIELD_NUMBER)) {
				setSingleString(person.getMetadataBuilder(), field);
			}

			final Map<String, MergedPerson> mergedMap = Maps.newHashMap();
			for(MergedPerson merged : person.getMergedpersonList()) {
				mergedMap.put(merged.getId(), merged);
			}
			person.clearMergedperson().addAllMergedperson(mergedMap.values());

			final Map<String, CoAuthor> coAuthorMap = Maps.newHashMap();
			for(CoAuthor coAuthor : person.getCoauthorList()) {
				coAuthorMap.put(coAuthor.getId(), coAuthor);
			}
			person.clearCoauthor().addAllCoauthor(coAuthorMap.values());

			break;
		case project:
			break;
		case publication:
			final Publication.Metadata.Builder pub = builder.getEntityBuilder().getPublicationBuilder().getMetadataBuilder();
			setTitle(pub);

			// for (String field : Lists.newArrayList("subject", "relevantdate")) {
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.SUBJECT_FIELD_NUMBER,
					Publication.Metadata.RELEVANTDATE_FIELD_NUMBER)) {
				setStructuredProperty(pub, field);
			}
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.DESCRIPTION_FIELD_NUMBER)) {
				setLongestStringField(pub, field);
			}
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.SOURCE_FIELD_NUMBER)) {
				setUniqueStringField(pub, field);
			}

			// remove the inner authors, rely on the children
			builder.getEntityBuilder().getPublicationBuilder().clearAuthor();
			break;
		case dataset:
			final Dataset.Metadata.Builder dataset = builder.getEntityBuilder().getDatasetBuilder().getMetadataBuilder();
			setTitle(dataset);

			// for (String field : Lists.newArrayList("subject", "relevantdate")) {
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.SUBJECT_FIELD_NUMBER,
					Publication.Metadata.RELEVANTDATE_FIELD_NUMBER)) {
				setStructuredProperty(dataset, field);
			}
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.DESCRIPTION_FIELD_NUMBER)) {
				setLongestStringField(dataset, field);
			}
			for (final String field : DNGFUtils.getFieldNames(Publication.Metadata.getDescriptor(), Publication.Metadata.SOURCE_FIELD_NUMBER)) {
				setUniqueStringField(dataset, field);
			}

			// remove the inner authors, rely on the children
			builder.getEntityBuilder().getDatasetBuilder().clearAuthor();
			break;
		default:
			break;
		}
		return builder;
	}

	/**
	 * Helper method, avoid duplicated StructuredProperties in the given builder for the given fieldName
	 *
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setStructuredProperty(final Builder builder, final String fieldName) {
		final Map<String, StructuredProperty> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StructuredProperty> sps = (List<StructuredProperty>) builder.getField(fd);

		if ((sps != null) && !sps.isEmpty()) {
			for (final StructuredProperty sp : sps) {
				map.put(sp.getValue(), sp);
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	/**
	 * Helper method, avoid duplicated KeyValues in the given builder for the given fieldName
	 *
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setKeyValues(final Builder builder, final String fieldName) {
		final Map<String, KeyValue> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<KeyValue> kvs = (List<KeyValue>) builder.getField(fd);

		if ((kvs != null) && !kvs.isEmpty()) {
			for (final KeyValue sp : kvs) {
				map.put(sp.getKey(), sp);
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setSingleString(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);
		if ((field != null) && !field.isEmpty()) {
			final StringField s = (StringField) Iterables.getLast(Iterables.filter(field, skipEmptyStringField), "");

			if ((s != null) && (s.getValue() != null) && !s.getValue().isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(s));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setLongestStringField(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);

		if ((field != null) && !field.isEmpty()) {
			final StringField.Builder max = StringField.newBuilder().setValue("");
			int maxLength = 0;
			for (final StringField sf : field) {
				if (sf.getValue().length() > maxLength) {
					maxLength = sf.getValue().length();
					max.clear();
					max.mergeFrom(sf);
				}
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(max.build()));
		}
	}

	@SuppressWarnings("unchecked")
	private void setUniqueStringField(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);
		final Map<String, StringField> map = Maps.newHashMap();
		if ((field != null) && !field.isEmpty()) {
			for (final StringField s : Iterables.filter(field, skipEmptyStringField)) {
				map.put(s.getValue(), s);
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
		}
	}

	@SuppressWarnings("unchecked")
	private void setUniqueString(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<String> field = (List<String>) builder.getField(fd);
		final Set<String> set = Sets.newHashSet();
		if ((field != null) && !field.isEmpty()) {
			for (final String s : Iterables.filter(field, skipEmptyString)) {
				set.add(s);
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(set));
		}
	}

	private void setTitle(final Publication.Metadata.Builder metadata) {
		final Iterable<StructuredProperty> filtered = Iterables.filter(metadata.getTitleList(), DNGFUtils.mainTitleFilter());

		if (!Iterables.isEmpty(filtered)) {
			metadata.clearTitle().addTitle(Iterables.getLast(filtered));
		}
	}

	private void setTitle(final Dataset.Metadata.Builder metadata) {
		final Iterable<StructuredProperty> filtered = Iterables.filter(metadata.getTitleList(), DNGFUtils.mainTitleFilter());

		if (!Iterables.isEmpty(filtered)) {
			metadata.clearTitle().addTitle(Iterables.getLast(filtered));
		}
	}

}
