package eu.dnetlib.data.hadoop.mapreduce;

import java.io.IOException;
import java.util.Date;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.mapred.Counters;
import org.apache.hadoop.mapred.Counters.Counter;
import org.apache.hadoop.mapred.Counters.Group;
import org.apache.hadoop.mapred.JobStatus;
import org.apache.hadoop.mapred.RunningJob;

import com.google.common.collect.Maps;

import eu.dnetlib.data.hadoop.blackboard.JobCompletion;
import eu.dnetlib.data.hadoop.blackboard.JobMonitor;
import eu.dnetlib.rmi.data.hadoop.HadoopServiceException;

public class MapreduceJobMonitor extends JobMonitor {

	private static final Log log = LogFactory.getLog(MapreduceJobMonitor.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final RunningJob runningJob;

	public MapreduceJobMonitor(final RunningJob runningJob, final JobCompletion callback) {
		super(callback);
		this.runningJob = runningJob;
	}

	@Override
	public void run() {
		try {
			log.info("waiting for job completion: " + getRunningJob().getID().getId());

			int runState = getRunState();
			while (!getRunningJob().isComplete()) {
				Thread.sleep(monitorSleepTimeSecs * 1000);

				// if the run state has changed we update the last activity
				final int currentState = getRunState();
				if (currentState != runState) {
					runState = currentState;
					lastActivity = new Date(System.currentTimeMillis());
				}
			}

			if (getRunningJob().isSuccessful()) {
				log.info("job successful: " + getRunningJob().getID().getId());
				getCallback().done(asMap(getRunningJob().getCounters()));
			} else {
				final String msg = "hadoop job: " + getHadoopId() + " failed with status: " + getStatus();
				getCallback().failed(msg, new HadoopServiceException(msg));
			}
		} catch (final Throwable e) {
			getCallback().failed(getHadoopId(), e);
		}
	}

	@Override
	public String getHadoopId() {
		return String.valueOf(getRunningJob().getID().getId());
	}

	// TODO: many hadoop classes are available under the newer package org.apache.hadoop.mapreduce.Counters,
	// but the JobClient and its returned objects are still using the old and deprecated org.apache.hadoop.mapred
	protected Map<String, String> asMap(final Counters counters) {
		final Map<String, String> res = Maps.newHashMap();
		if (counters != null) {
			for (final Group group : counters) {
				for (final Counter counter : group) {
					res.put(group.getDisplayName() + "." + counter.getDisplayName(), String.valueOf(counter.getValue()));
				}
			}
		}
		return res;
	}

	public RunningJob getRunningJob() {
		return runningJob;
	}

	@Override
	public String getStatus() {
		try {
			return JobStatus.getJobRunState(getRunState());
		} catch (final IOException e) {
			log.error("error accessing job status", e);
			return "UNKNOWN";
		}
	}

	private int getRunState() throws IOException {
		return getRunningJob().getJobStatus().getRunState();
	}

	@Override
	public Date getLastActivity() {
		return lastActivity;
	}

	@Override
	public Date getStartTime() throws HadoopServiceException {
		try {
			return new Date(getRunningJob().getJobStatus().getStartTime());
		} catch (final IOException e) {
			throw new HadoopServiceException("unable to read job start time", e);
		}
	}

	@Override
	public String getTrackerUrl() {
		return getRunningJob().getTrackingURL();
	}

	@Override
	public void kill() {
		try {
			log.info("killing job: " + getHadoopId());
			getRunningJob().killJob();
		} catch (final IOException e) {
			log.error("unable to kill job: " + getHadoopId(), e);
		}
	}

}
