package eu.dnetlib.enabling.is.lookup;

import java.io.StringReader;
import java.util.List;
import javax.jws.WebService;

import eu.dnetlib.DnetConstants;
import eu.dnetlib.enabling.is.store.ISStore;
import eu.dnetlib.enabling.is.store.ISStoreException;
import eu.dnetlib.enabling.tools.AbstractBaseService;
import eu.dnetlib.enabling.tools.CompatResourceIdentifierResolverImpl;
import eu.dnetlib.enabling.tools.ResourceIdentifierResolver;
import eu.dnetlib.enabling.tools.XQueryUtils;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

/**
 * ISLookUpService implementation.
 * 
 * @author marko
 * @author michele
 * 
 */
@WebService(targetNamespace = "http://services.dnetlib.eu/")
public class ISLookUpServiceImpl extends AbstractBaseService implements ISLookUpService { // NOPMD

	/**
	 * error message when the given collection cannot be fetched.
	 */
	private static final String COLLECTION_ERROR = "cannot get collection";

	/**
	 * base xmldb directory.
	 */
	private static final String DB_BASE_DIR = "/db/DRIVER";

	/**
	 * error message when the profile is not present in the db.
	 */
	private static final String PROFILE_NOT_FOUND = "Profile not found";

	private ISStore isStore;

	// NOPMD by marko on 11/24/08
	// 5:02 PM
	/**
	 * logger.
	 */
	public static final Log log = LogFactory.getLog(ISLookUpServiceImpl.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * resource identifier resolver. Resolves identifiers to xmldb file and collection names.
	 */
	private ResourceIdentifierResolver resIdManager = new CompatResourceIdentifierResolverImpl();

	/**
	 * xquery utils. Used to obtain the xmldb collection mapping for resources.
	 */
	private XQueryUtils xqueryUtils;


	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#retrieveCollection(java.lang.String)
	 */
	@Override
	public String retrieveCollection(final String profId) throws ISLookUpException {
		try {
			String profile = this.getResourceProfile(profId);
			final List<String> list = quickSearchProfile("for $x in collection('/db/DRIVER/CollectionDSResources') where $x//FATHER/@id = '" + profId
					+ "' return $x//RESOURCE_IDENTIFIER/@value/string()");

			if (!list.isEmpty()) {
				final SAXReader reader = new SAXReader();
				final Document doc = reader.read(new StringReader(profile));
				final Element childrenNode = (Element) doc.selectSingleNode("//CHILDREN"); // NOPMD
				for (final String idC : list) {
					childrenNode.addElement("CHILD").addAttribute("id", idC);
				}
				profile = doc.asXML();
			}

			return profile;
		} catch (final Exception e) {
			throw new ISLookUpException(e);
		}
	}


	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#getResourceProfile(java.lang.String)
	 */
	@Override
	public String getResourceProfile(final String profId) throws ISLookUpException {
		if (profId == null || profId.isEmpty()) { throw new ISLookUpException("Invalid null profile ID: " + profId); }

		try {
			final String res = isStore.getXML(getFileNameForId(profId),
					xqueryUtils.getRootCollection() + getFileCollForId(profId));
			if (res == null) { throw new ISLookUpDocumentNotFoundException("document " + profId + " not found"); }
			return res;
		} catch (final ISStoreException e) {
			throw new ISLookUpException(e);
		}
	}

	/**
	 * obtain the xmldb file name from the profile identifier.
	 * 
	 * @param profId
	 *            profile id
	 * @return xml db file name
	 */
	String getFileNameForId(final String profId) {
		return resIdManager.getFileName(profId);
	}

	/**
	 * obtain the xmldb collection name from the profile identifier.
	 * 
	 * @param profId
	 *            profile id
	 * @return plaintext xmldb collection name
	 */
	String getFileCollForId(final String profId) {
		return resIdManager.getCollectionName(profId);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#getResourceProfileByQuery(java.lang.String)
	 */
	@Override
	public String getResourceProfileByQuery(final String xquery) throws ISLookUpException {
		String resource;
		try {
			resource = isStore.getXMLbyQuery(xquery);
			if (resource == null || resource.isEmpty()) { throw new ISLookUpDocumentNotFoundException(PROFILE_NOT_FOUND); }
			return resource;
		} catch (final ISStoreException e) {
			throw new ISLookUpException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#getResourceQoSParams(java.lang.String)
	 */
	@Override
	public String getResourceQoSParams(final String profId) throws ISLookUpException {
		final String query = "for $x in collection('/db/DRIVER/ServiceResources') where $x//RESOURCE_IDENTIFIER/@value = '" + profId + "' return $x//QOS";
		return getResourceProfileByQuery(query);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#getResourceTypeSchema(java.lang.String)
	 */
	@Override
	public String getResourceTypeSchema(final String resourceType) throws ISLookUpException {
		if (resourceType == null || resourceType.isEmpty()) { throw new ISLookUpException("Invalid resourceType"); }

		try {
			return isStore.getXML(resourceType, xqueryUtils.getRootCollection() + DnetConstants.RESOURCE_TYPES);
		} catch (final ISStoreException e) {
			throw new ISLookUpException(e);
		}
	}




	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#listResourceTypes()
	 */
	@Override
	public List<String> listResourceTypes() throws ISLookUpException {
		try {
			return isStore.getFileNames(DB_BASE_DIR + "/" + DnetConstants.RESOURCE_TYPES);
		} catch (final ISStoreException e) {
			throw new ISLookUpException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.rmi.enabling.ISLookUpService#quickSearchProfile(java.lang.String)
	 */
	@Override
	public List<String> quickSearchProfile(final String xquery) throws ISLookUpException {
		try {
			return isStore.quickSearchXML(xquery);
		} catch (final ISStoreException e) {
			throw new ISLookUpException(e);
		}
	}

	public ResourceIdentifierResolver getResIdManager() {
		return resIdManager;
	}

	public void setResIdManager(final ResourceIdentifierResolver resIdManager) {
		this.resIdManager = resIdManager;
	}

	@Required
	public void setXqueryUtils(final XQueryUtils xqueryUtils) {
		this.xqueryUtils = xqueryUtils;
	}

	public XQueryUtils getXqueryUtils() {
		return xqueryUtils;
	}

	public ISStore getIsStore() {
		return isStore;
	}

	@Required
	public void setIsStore(final ISStore isStore) {
		this.isStore = isStore;
	}

}
