package eu.dnetlib.enabling.inspector;

import java.io.File;
import java.io.IOException;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.xpath.XPathExpressionException;

import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.tools.OpaqueResource;
import eu.dnetlib.enabling.tools.ResourceIdentifierResolver;
import eu.dnetlib.enabling.tools.StringOpaqueResource;
import eu.dnetlib.rmi.enabling.ISLookUpException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.enabling.ISRegistryException;
import eu.dnetlib.rmi.enabling.ISRegistryService;
import eu.dnetlib.xml.database.XMLDatabase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.xml.sax.SAXException;
import org.xmldb.api.base.XMLDBException;

/**
 * Implement mid-level functionality for validation and invalidation of profiles browsed with the ResourceTreeController.
 * 
 * @see ResourceTreeController
 * @author marko
 * 
 */
@Controller
public class ValidationController extends AbstractInspectorController {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(ValidationController.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * base index.do path.
	 */
	private static final String INDEX_DO = "/inspector/index.do";

	/**
	 * xml database.
	 */
	@Resource(name = "existDatabase")
	private transient XMLDatabase xmlDatabase;

	/**
	 * service locator.
	 */
	@Autowired
	private UniqueServiceLocator serviceLocator;

	/**
	 * resolve xmldb ids to resource ids.
	 */
	@Autowired
	private transient ResourceIdentifierResolver resIdResolver;

	@Autowired
	private TransformerFactory saxonTransformerFactory;

	/**
	 * high level: profile validation via registry.
	 * 
	 * @param model
	 *            model
	 * @param request
	 *            request
	 * @return view name
	 * @throws XMLDBException
	 *             happens
	 * @throws ParserConfigurationException
	 *             shouldn't happen
	 * @throws IOException
	 *             shouldn't happen
	 * @throws SAXException
	 *             shouldn't happen
	 * @throws XPathExpressionException
	 *             shouldn't happen
	 * @throws ISRegistryException
	 *             happens
	 * @throws ISLookUpException
	 *             happens
	 */
	@RequestMapping("/inspector/index.do/**/validate")
	String validate(final Model model, final HttpServletRequest request) throws XMLDBException, XPathExpressionException, SAXException, IOException,
			ParserConfigurationException, ISRegistryException, ISLookUpException {

		final String path = request.getPathInfo().replace(INDEX_DO, "").replace("/validate", "");

		final File fileHelper = new File(path);
		final String collection = fileHelper.getParent();
		final String fileName = fileHelper.getName();

		log.info("validating: " + path);
		final OpaqueResource resource = new StringOpaqueResource(xmlDatabase.read(fileName, collection), saxonTransformerFactory);
		log.info("validating a " + resource.getResourceType());

		final String newId = serviceLocator.getService(ISRegistryService.class, true).validateProfile(resource.getResourceId());
		final OpaqueResource valid = new StringOpaqueResource(serviceLocator.getService(ISLookUpService.class).getResourceProfile(newId), saxonTransformerFactory);

		final String vFileName = resIdResolver.getFileName(valid.getResourceId());
		final String vCollectionName = resIdResolver.getCollectionName(valid.getResourceId());

		return "redirect:../../../" + vCollectionName + "/" + vFileName + "/show";
	}

	/**
	 * high level: profile invalidation via registry.
	 * 
	 * @param model
	 *            model
	 * @param request
	 *            request
	 * @return view name
	 * @throws XMLDBException
	 *             happens
	 * @throws ParserConfigurationException
	 *             shouldn't happen
	 * @throws IOException
	 *             shouldn't happen
	 * @throws SAXException
	 *             shouldn't happen
	 * @throws XPathExpressionException
	 *             shouldn't happen
	 * @throws ISRegistryException
	 *             happens
	 * @throws ISLookUpException
	 *             happens
	 */
	@RequestMapping("/inspector/index.do/**/invalidate")
	String invalidate(final Model model, final HttpServletRequest request) throws XMLDBException, XPathExpressionException, SAXException, IOException,
			ParserConfigurationException, ISRegistryException, ISLookUpException {

		final String path = request.getPathInfo().replace(INDEX_DO, "").replace("/invalidate", "");

		final File fileHelper = new File(path);
		final String collection = fileHelper.getParent();
		final String fileName = fileHelper.getName();

		log.info("invalidating: " + path);
		final OpaqueResource resource = new StringOpaqueResource(xmlDatabase.read(fileName, collection), saxonTransformerFactory);
		log.info("invalidating a " + resource.getResourceType());

		final String newId = serviceLocator.getService(ISRegistryService.class, true).invalidateProfile(resource.getResourceId());
		final OpaqueResource invalid = new StringOpaqueResource(serviceLocator.getService(ISLookUpService.class).getResourceProfile(newId), saxonTransformerFactory);

		final String vFileName = resIdResolver.getFileName(invalid.getResourceId());
		final String vCollectionName = resIdResolver.getCollectionName(invalid.getResourceId());

		return "redirect:../../../" + vCollectionName + "/" + vFileName + "/show";
	}

}
