package eu.dnetlib.data.mdstore.plugins;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.collect.Sets;

import eu.dnetlib.data.db.DbGroup;
import eu.dnetlib.data.db.DbPerson;
import eu.dnetlib.data.db.DbPersonsDao;
import eu.dnetlib.data.mdstore.plugins.objects.Affiliation;
import eu.dnetlib.data.mdstore.plugins.objects.CnrPerson;
import eu.dnetlib.data.mdstore.plugins.objects.MdRecord;
import eu.dnetlib.data.utils.XsltFunctions;

public class EnrichLabsPlugin extends MdRecordPlugin {

	private static final Log log = LogFactory.getLog(EnrichLabsPlugin.class);

	public static final DbGroup UNKNOWN_GROUP = new DbGroup("UNKNOWN", "UNKNOWN", "UNKNOWN");

	public static final String DNET_PERSON_PREFIX = "dnet:pers:";

	public static final int ISTI_LABS_START_YEAR = 2002;

	@Autowired
	private DbPersonsDao dao;

	private Map<String, String> masters = new HashMap<>();
	private Map<String, DbPerson> affiliations = new HashMap<>();
	private Set<String> newRegistered = new HashSet<>();

	@Override
	protected void reconfigure(final Map<String, String> params) {
		newRegistered.clear();
		masters = dao.obtainRawPersonToMasterRels();
		affiliations = dao.listPersonsWithAffiliations().stream().collect(Collectors.toMap(DbPerson::getId, a -> a));
	}

	@Override
	protected void resetConfiguration() {
		affiliations.clear();
	}

	@Override
	protected boolean updateRecord(final String recordId, final MdRecord doc) {
		final int year = doc.getDate();

		final Set<CnrPerson> newCnrPersons = new LinkedHashSet<>();

		for (final CnrPerson cp : doc.getCnrPersons()) {
			cp.getAffiliations().clear();

			final Pattern pattern = Pattern.compile("info:cnr-pdr\\/author\\/(.+)\\/(.+)\\/(.+)");
			final Matcher matcher = pattern.matcher(cp.getId());

			if (matcher.find()) {
				processPerson(cp, matcher.group(1), matcher.group(2), matcher.group(3), null, null, year);
			} else if (StringUtils.isNotBlank(cp.getName())) {
				processPerson(cp, cp.getId(), cp.getName(), "", null, cp.getOrcid(), year);
			}

			if (StringUtils.isNotBlank(cp.getOrcid())) {
				doc.getCreators()
						.stream()
						.filter(c -> isSamePerson(cp.getName(), c.getName()))
						.filter(c -> StringUtils.isBlank(c.getOrcid()))
						.findFirst()
						.ifPresent(c -> c.setOrcid(cp.getOrcid()));
			}

			if (!cp.getAffiliations().isEmpty()) {
				newCnrPersons.add(cp);
			}
		}
		doc.setCnrPersons(newCnrPersons);

		return true;
	}

	private void processPerson(final CnrPerson cp,
			final String code,
			final String surname,
			final String name,
			final String suffix,
			final String orcid,
			final int year) {

		if (newRegistered.contains(code)) {
			// to avoid a repeated insertion of a new person
			return;
		}

		final DbPerson p = code.startsWith(DNET_PERSON_PREFIX) ? affiliations.get(code) : affiliations.get(masters.get(code));

		if (p != null) {
			cp.setId(p.getId());
			cp.setName(p.getFullname());
			cp.setOrcid(p.getOrcid());

			if (year >= ISTI_LABS_START_YEAR) {
				final Set<DbGroup> groups = p.getGroups().get(year);
				if (groups != null) {
					for (final DbGroup g : groups) {
						if (!g.getId().equals(UNKNOWN_GROUP.getId())) {
							cp.getAffiliations().add(new Affiliation(p.getId(), g.getType(), g.getId(), g.getName()));
						}
					}
				} else {
					log.warn("Affiliation not found, id: " + p.getId() + ", year: " + year);
					dao.registerAffiliation(p.getId(), UNKNOWN_GROUP.getId(), year);
					p.getGroups().put(year, Sets.newHashSet(UNKNOWN_GROUP));
				}
			}
		} else {
			log.warn("Person not found, code: " + code);
			dao.registerRawPerson(code, XsltFunctions.capitalize(name), XsltFunctions.capitalize(surname), suffix, orcid);
			newRegistered.add(code);
		}
	}

	protected static boolean isSamePerson(final String fullname, final String other) {

		final String surname = StringUtils.substringBefore(fullname, ",").trim().toLowerCase();
		final String name = StringUtils.substringAfter(fullname, ",").trim().toLowerCase();

		final List<Character> initials1 = Arrays.stream(name.split(" "))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(s -> s.charAt(0))
				.collect(Collectors.toList());

		if (initials1.isEmpty()) { return false; }

		if (!other.toLowerCase().startsWith(surname)) { return false; }

		final String[] parts = StringUtils.substringAfter(other.toLowerCase(), surname)
				.replaceAll("\\,", " ")
				.replaceAll("\\.", " ")
				.split(" ");

		final List<Character> initials2 = Arrays.stream(parts)
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(s -> s.charAt(0))
				.collect(Collectors.toList());

		if (initials2.isEmpty()) {
			return false;
		} else if (initials1.containsAll(initials2)) {
			return true;
		} else if (initials2.containsAll(initials1)) {
			return true;
		} else {
			return false;
		}

	}

}
