package eu.dnetlib.data.mdstore.plugins.objects;

import java.io.StringReader;
import java.net.URI;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import eu.dnetlib.data.utils.HttpFetcher;

@XmlAccessorType(XmlAccessType.FIELD)
public class Project {

	@XmlElement(name = "infoId")
	private String infoId;
	@XmlElement(name = "openaireId")
	private String openaireId;
	@XmlElement(name = "code")
	private String code;
	@XmlElement(name = "name")
	private String name;
	@XmlElement(name = "acronym")
	private String acronym;
	@XmlElement(name = "funder")
	private String funder;
	@XmlElement(name = "program")
	private String program;
	@XmlElement(name = "jurisdiction")
	private String jurisdiction;

	private static final Log log = LogFactory.getLog(Project.class);

	private static final String PROJECT_REGEX = "info:eu-repo\\/grantAgreement\\/(.*)\\/(.*)\\/(.*)\\/(.*)\\/(.*)\\/(.*)";

	public static Project newInstance(final URI url) {

		try {
			final SAXReader reader = new SAXReader();
			final String s = HttpFetcher.fetch(url);
			final Document doc = reader.read(new StringReader(s));

			final String openaireId = doc.valueOf("//*[local-name()='objIdentifier']");

			if (StringUtils.isBlank(openaireId)) {
				log.warn("Project not found using OpenAIRE API, url: " + url);
				return null;
			}

			final String code = doc.valueOf("//code");
			final String name = doc.valueOf("//title");
			final String acronym = doc.valueOf("//acronym");
			final String funder = doc.valueOf("//funder/shortname");
			final String program = doc.valueOf("//funding_level_0/name");
			final String jurisdiction = doc.valueOf("//funder/jurisdiction");

			return new Project(openaireId, code, name, acronym, funder, program, jurisdiction);
		} catch (final DocumentException e) {
			log.error("Error parsing document from url " + url);
			throw new RuntimeException(e);
		}

	}

	public static Project newInstance(final String infoId) {
		final Pattern pattern = Pattern.compile(PROJECT_REGEX);
		final Matcher matcher = pattern.matcher(infoId);
		if (matcher.find()) {
			final String openaireId = "";
			final String funder = matcher.group(1).replaceAll("%2F", "/");
			final String program = matcher.group(2).replaceAll("%2F", "/");
			final String code = matcher.group(3).replaceAll("%2F", "/");
			final String jurisdiction = matcher.group(4).replaceAll("%2F", "/");
			final String name = StringUtils.defaultIfBlank(matcher.group(5).replaceAll("%2F", "/"), funder + "/" + program + "/" + code);
			final String acronym = StringUtils.defaultIfBlank(matcher.group(6), name).replaceAll("%2F", "/");

			if (StringUtils.isNotEmpty(code) && StringUtils.isNotEmpty(program)
					&& StringUtils.isNotEmpty(funder)) {
				return new Project(openaireId, code, name, acronym, funder, program, jurisdiction);
			}
		}

		log.warn("Invalid project ID: " + infoId);
		return null;

	}

	public static boolean isValid(final String infoId) {
		return Project.newInstance(infoId) != null;
	}

	public Project() {}

	public Project(final String openaireId, final String code, final String name, final String acronym, final String funder, final String program,
			final String jurisdiction) {
		this.openaireId = openaireId;
		this.code = code;
		this.name = name;
		this.acronym = acronym;
		this.funder = funder;
		this.program = program;
		this.jurisdiction = jurisdiction;
	}

	public String getOpenaireId() {
		return openaireId;
	}

	public String getCode() {
		return code;
	}

	public String getName() {
		return name;
	}

	public String getAcronym() {
		return acronym;
	}

	public String getFunder() {
		return funder;
	}

	public String getProgram() {
		return program;
	}

	public String getJurisdiction() {
		return jurisdiction;
	}

	public boolean match(final String infoId) {
		final Project p = Project.newInstance(infoId);
		return ((p != null) && funder.equals(p.funder) && program.equals(p.program) && code.equals(p.code));
	}

	@Override
	public String toString() {
		return "Project [openaireId=" + openaireId + ", code=" + code + ", name=" + name + ", acronym=" + acronym + ", funder=" + funder + ", program="
				+ program + ", jurisdiction=" + jurisdiction + "]";
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = (prime * result) + ((code == null) ? 0 : code.hashCode());
		result = (prime * result) + ((funder == null) ? 0 : funder.hashCode());
		result = (prime * result) + ((program == null) ? 0 : program.hashCode());
		return result;
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) { return true; }
		if (obj == null) { return false; }
		if (getClass() != obj.getClass()) { return false; }
		final Project other = (Project) obj;
		if (code == null) {
			if (other.code != null) { return false; }
		} else if (!code.equals(other.code)) { return false; }
		if (funder == null) {
			if (other.funder != null) { return false; }
		} else if (!funder.equals(other.funder)) { return false; }
		if (program == null) {
			if (other.program != null) { return false; }
		} else if (!program.equals(other.program)) { return false; }
		return true;
	}

	public String getInfoId() {
		return infoId;
	}

	public void setInfoId(final String infoId) {
		this.infoId = infoId;
	}

	public void setOpenaireId(final String openaireId) {
		this.openaireId = openaireId;
	}

	public void setCode(final String code) {
		this.code = code;
	}

	public void setName(final String name) {
		this.name = name;
	}

	public void setAcronym(final String acronym) {
		this.acronym = acronym;
	}

	public void setFunder(final String funder) {
		this.funder = funder;
	}

	public void setProgram(final String program) {
		this.program = program;
	}

	public void setJurisdiction(final String jurisdiction) {
		this.jurisdiction = jurisdiction;
	}
}
