package eu.dnetlib.data.mdstore.plugins;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import eu.dnetlib.data.mdstore.modular.MDStoreFeeder;
import eu.dnetlib.data.mdstore.modular.action.DoneCallback;
import eu.dnetlib.data.mdstore.modular.action.FailedCallback;
import eu.dnetlib.data.mdstore.modular.action.MDStorePlugin;
import eu.dnetlib.data.mdstore.modular.connector.MDStore;
import eu.dnetlib.data.mdstore.modular.connector.MDStoreDao;
import eu.dnetlib.data.mdstore.modular.mongodb.MDStoreDaoImpl;
import eu.dnetlib.data.mdstore.modular.mongodb.MongoMDStore;
import eu.dnetlib.rmi.data.MDStoreServiceException;

public abstract class AbstractIstiMDStorePlugin implements MDStorePlugin {

	public MDStoreFeeder getFeeder() {
		return feeder;
	}

	public void setFeeder(final MDStoreFeeder feeder) {
		this.feeder = feeder;
	}

	private MDStoreDaoImpl dao;

	@Autowired
	private MDStoreFeeder feeder;

	private static final Log log = LogFactory.getLog(AbstractIstiMDStorePlugin.class);

	@Override
	public final void run(final MDStoreDao dao, final Map<String, String> params, final DoneCallback doneCallback, final FailedCallback failedCallback) {

		if (dao instanceof MDStoreDaoImpl) {
			setDao((MDStoreDaoImpl) dao);
		} else {
			log.error("a mongo mdStore service is required");
			failedCallback.call(new MDStoreServiceException("a mongo mdStore service is required"));
		}

		final LocalDateTime start = LocalDateTime.now();
		try {
			process(resolveStore(params, "mdId"), params);
			log.info("Plugin Execution Time: " + ChronoUnit.SECONDS.between(start, LocalDateTime.now()));
			doneCallback.call(params);
		} catch (final Throwable e) {
			log.error(e.getMessage(), e);
			failedCallback.call(e);
		}
	}

	protected abstract void process(final MongoMDStore store, final Map<String, String> params) throws MDStoreServiceException;

	protected MongoMDStore resolveStore(final Map<String, String> params, final String key) throws MDStoreServiceException {
		final String mdId = params.get(key);

		if (StringUtils.isBlank(mdId)) { throw new MDStoreServiceException("mdID is empty"); }

		final MDStore store = dao.getMDStore(mdId);

		if (store == null) {
			log.error("mdStore not found, id: " + mdId);
			throw new MDStoreServiceException("mdStore not found, id: " + mdId);
		} else if (store instanceof MongoMDStore) {
			return (MongoMDStore) store;
		} else {
			log.error("a mongo mdStore is required");
			throw new MDStoreServiceException("a mongo mdStore is required");
		}
	}

	protected void touch(final MongoMDStore store) throws MDStoreServiceException {
		feeder.touch(store.getId(), dao.refreshSize(store.getId()));
	}

	public MDStoreDaoImpl getDao() {
		return dao;
	}

	public void setDao(final MDStoreDaoImpl dao) {
		this.dao = dao;
	}

}
