package eu.dnetlib.msro.workflows.nodes;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.Node;
import org.dom4j.QName;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.enabling.resultset.factory.ResultSetFactory;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.common.ResultSet;
import eu.dnetlib.rmi.manager.MSROException;

public class PatchCinecaIdsJobNode extends SimpleJobNode {

	@Autowired
	private ResultSetFactory resultSetFactory;

	private Resource idMapFile;

	private String inputEprParam;
	private String outputEprParam;

	public static final String cinecaOaiPrefix = "oai:iris.cnr.it:";
	public static final String peopleOaiPrefix = "oai:it.cnr:prodotti:";
	public static final String peopleNamespacePrefix = "people______::";

	private static final Namespace ISTI_NS = new Namespace("isti", "http://openportal.isti.cnr.it");
	private static final Namespace DRI_NS = new Namespace("dri", "http://www.driver-repository.eu/namespace/dri");

	private static final Log log = LogFactory.getLog(PatchCinecaIdsJobNode.class);

	@Override
	protected String execute(final Env env) throws Exception {

		@SuppressWarnings("unchecked")
		final ResultSet<String> rsIn = env.getAttribute(inputEprParam, ResultSet.class);

		if (rsIn == null) { throw new MSROException("Passed rs is null"); }

		final Map<String, String> idMap = loadMap(idMapFile);

		log.info("potential mapped ids: " + idMap.size());

		final ResultSet<String> rsOut = resultSetFactory.map(rsIn, String.class, xml -> patchId(xml, idMap));

		env.setAttribute(outputEprParam, rsOut);

		return Arc.DEFAULT_ARC;
	}

	protected Map<String, String> loadMap(final Resource mapResource) throws IOException {

		final Gson gson = new Gson();

		final TypeToken<HashMap<String, String>> tt = new TypeToken<HashMap<String, String>>() {};

		try (InputStream is = mapResource.getInputStream(); InputStreamReader reader = new InputStreamReader(is)) {
			return gson.fromJson(reader, tt.getType());
		}

	}

	protected String patchId(final String xml, final Map<String, String> idMap) {
		try {

			final Document doc = DocumentHelper.parseText(xml);

			final Element header = (Element) doc.selectSingleNode("//*[local-name() = 'header']");
			for (final Object o : header.selectNodes("./*[local-name() = 'identifier']")) {
				final Node n = (Node) o;
				final String oldValue = n.getText().trim();

				if (oldValue.startsWith(cinecaOaiPrefix)) {
					final String peopleId = idMap.get(StringUtils.substringAfter(oldValue, cinecaOaiPrefix));

					if (StringUtils.isNotBlank(peopleId)) {
						final String newValue = peopleOaiPrefix + peopleId;
						n.setText(newValue);
						header.addElement(new QName("objIdentifier", DRI_NS)).setText(peopleNamespacePrefix + Hashing.md5(newValue));
						header.addElement(new QName("hiddenIdentifier", ISTI_NS)).setText(oldValue);
						log.debug("* MAPPED ID: " + oldValue + " -> " + newValue);
					}
				}
			}

			return doc.asXML();
		} catch (final Exception e) {
			throw new RuntimeException(e);
		}
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getOutputEprParam() {
		return outputEprParam;
	}

	public void setOutputEprParam(final String outputEprParam) {
		this.outputEprParam = outputEprParam;
	}

	public Resource getIdMapFile() {
		return idMapFile;
	}

	@Required
	public void setIdMapFile(final Resource idMapFile) {
		this.idMapFile = idMapFile;
	}

}
