package eu.dnetlib.data.mdstore.plugins;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;

import eu.dnetlib.data.mdstore.plugins.objects.MdRecord;
import eu.dnetlib.data.mdstore.plugins.objects.Project;

public class EnrichProjectsPlugin extends MdRecordPlugin {

	private static final Log log = LogFactory.getLog(EnrichProjectsPlugin.class);

	@Value("${plugin.enrich.projects.openaire.url}")
	private String baseUrl;

	private Map<String, Project> mapProjects = new HashMap<>();

	@Override
	protected void reconfigure(final Map<String, String> params) {
		log.debug("Cleaning projects cache");
		mapProjects.clear();
	}

	@Override
	protected void resetConfiguration() {
		log.debug("Cleaning projects cache");
		mapProjects.clear();
	}

	@Override
	protected boolean updateRecord(final String recordId, final MdRecord record) {
		log.debug("Found " + record.getProjects().size() + " projects");

		if (record.getProjects().isEmpty()) { return false; }

		final Set<Project> list = new LinkedHashSet<>();
		for (final Project p : record.getProjects()) {
			if (StringUtils.isNotBlank(p.getInfoId())) {
				list.add(findEnrichedProject(Project.newInstance(p.getInfoId())));
			} else if (allValidParams(p.getFunder(), p.getProgram(), p.getCode())) {
				list.add(findEnrichedProject(p));
			} else {
				list.add(p);
			}
		}

		log.debug("Processed " + list.size() + " projects");

		if (list.isEmpty()) { return false; }

		record.getProjects().clear();
		record.getProjects().addAll(list);

		return true;
	}

	private boolean allValidParams(final String... params) {
		for (final String s : params) {
			if (StringUtils.isBlank(s) || s.trim().contains(" ")) { return false; }
		}
		return true;
	}

	private Project findEnrichedProject(final Project simpleProject) {
		final String s = simpleProject.getFunder() + "/" + simpleProject.getProgram() + "/" + simpleProject.getCode();
		if (!mapProjects.containsKey(s)) {
			final Project project = resolveProject(s);
			mapProjects.put(s, project != null ? project : simpleProject);
		}
		return mapProjects.get(s);
	}

	private Project resolveProject(final String s) {
		try {
			final String[] arr = s.split("/");
			final String url = String.format(baseUrl, arr[0], arr[1], arr[2]);
			return Project.newInstance(new URI(url));
		} catch (final URISyntaxException e) {
			log.error("Error resolving project: " + s, e);
			throw new RuntimeException("Error resolving project: " + s, e);
		}
	}

}
