package eu.dnetlib.data.mdstore.plugins;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.mongodb.MongoClient;
import com.mongodb.client.MongoCollection;

import eu.dnetlib.data.mdstore.plugins.objects.Dataset;
import eu.dnetlib.data.mdstore.plugins.objects.MdRecord;
import eu.dnetlib.data.mdstore.plugins.objects.dli.DliEntity;
import eu.dnetlib.data.mdstore.plugins.objects.dli.DliIdentifier;
import eu.dnetlib.data.mdstore.plugins.objects.dli.DliRelation;

public class EnrichDatasetsPlugin extends GenericDoiMdstorePlugin {

	private static final Log log = LogFactory.getLog(EnrichDatasetsPlugin.class);

	private static final Gson gson = new Gson();

	@Value("${plugin.enrich.dataset.dli.url}")
	private String baseUrl;

	@Autowired
	private MongoClient mongoClient;

	private Counter enrichDatasetsCounter = new Counter();

	@Override
	protected URI prepareURI(final String doi) throws URISyntaxException {
		return new URI(String.format(baseUrl, doi));
	}

	@Override
	protected void reconfigure(final Map<String, String> params) {
		enrichDatasetsCounter.reset();
	}

	@Override
	protected void resetConfiguration() {
		log.info("***** Openaire Enrichment - datasets  : " + enrichDatasetsCounter);
		enrichDatasetsCounter.reset();
	}

	@Override
	protected MongoCollection<org.bson.Document> getCacheCollection() {
		return mongoClient.getDatabase("API_CACHES").getCollection("DATASETS_API_CACHE");
	}

	@Override
	protected boolean updateDocument(final MdRecord doc, final String response) {

		doc.getDatasets().clear();

		final List<Dataset> datasets = new ArrayList<>();

		final List<DliRelation> rels = gson.fromJson(response, new TypeToken<List<DliRelation>>() {}.getType());

		for (final DliRelation rel : rels) {
			datasets.addAll(findRelatedDatasets(rel.getSource()));
			datasets.addAll(findRelatedDatasets(rel.getTarget()));
		}

		if (datasets.isEmpty()) {
			return false;
		} else {
			doc.getDatasets().addAll(datasets);
			enrichDatasetsCounter.incrementAfter(doc.getDatasets().size());
			return true;
		}
	}

	private List<Dataset> findRelatedDatasets(final DliEntity entity) {
		final List<Dataset> res = new ArrayList<>();
		if (entity.getObjectType().equalsIgnoreCase("dataset")) {
			final String title = entity.getTitle();
			for (final DliIdentifier id : entity.getIdentifiers()) {
				if (id.getSchema().equalsIgnoreCase("doi") && StringUtils.isNoneBlank(id.getIdentifier()) && StringUtils.isNotBlank(title)) {
					res.add(new Dataset(id.getIdentifier(), title));
				}
			}
		}
		return res;
	}

}
