package eu.dnetlib.data.mdstore.plugins;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import eu.dnetlib.data.db.DbGroup;
import eu.dnetlib.data.db.DbPerson;
import eu.dnetlib.data.db.DbPersonsDao;
import eu.dnetlib.data.mdstore.plugins.objects.CnrPerson;
import eu.dnetlib.data.mdstore.plugins.objects.MdRecord;
import eu.dnetlib.data.utils.XsltFunctions;

public class EnrichCnrPersonsPlugin extends MdRecordPlugin {

	private static final Log log = LogFactory.getLog(EnrichCnrPersonsPlugin.class);

	public static final DbGroup UNKNOWN_GROUP = new DbGroup("UNKNOWN", "UNKNOWN", "UNKNOWN");

	public static final String DNET_PERSON_PREFIX = "dnet:pers:";

	@Autowired
	private DbPersonsDao dao;

	private Map<String, String> masters = new HashMap<>();
	private Map<String, DbPerson> affiliations = new HashMap<>();
	private Set<String> newRegistered = new HashSet<>();

	@Override
	protected void reconfigure(final Map<String, String> params) {
		final String affiliation = params.get("affiliation");

		newRegistered.clear();
		masters = dao.obtainRawPersonToMasterRels();

		affiliations = dao.listPersonsWithAffiliations()
				.stream()
				.filter(p -> {
					for (final Set<DbGroup> groups : p.getGroups().values()) {
						for (final DbGroup g : groups) {
							if (g.getId().equals(affiliation)) { return true; }
						}
					}
					return false;
				})
				.collect(Collectors.toMap(DbPerson::getId, a -> a));
	}

	@Override
	protected void resetConfiguration() {
		masters.clear();
		affiliations.clear();
	}

	@Override
	protected boolean updateRecord(final String recordId, final MdRecord doc) {

		final Set<CnrPerson> newCnrPersons = new LinkedHashSet<>();

		for (final CnrPerson old : doc.getCnrPersons()) {
			final Pattern pattern = Pattern.compile("info:cnr-pdr\\/author\\/(.+)\\/(.+)\\/(.+)");
			final Matcher matcher = pattern.matcher(old.getId());

			final CnrPerson ncp;
			if (matcher.find()) {
				ncp = processPerson(matcher.group(1), matcher.group(2), matcher.group(3), null, null);
			} else if (StringUtils.isNotBlank(old.getName())) {
				ncp = processPerson(old.getId(), old.getName(), "", null, old.getOrcid());
			} else {
				ncp = null;
			}

			if (ncp != null) {
				if (StringUtils.isNotBlank(ncp.getOrcid())) {
					doc.getCreators()
							.stream()
							.filter(c -> isSamePerson(ncp.getName(), c.getName()))
							.filter(c -> StringUtils.isBlank(c.getOrcid()))
							.findFirst()
							.ifPresent(c -> c.setOrcid(ncp.getOrcid()));
				}

				newCnrPersons.add(ncp);
			}
		}

		doc.setCnrPersons(newCnrPersons);

		return true;
	}

	private CnrPerson processPerson(final String code,
			final String surname,
			final String name,
			final String suffix,
			final String orcid) {

		if (newRegistered.contains(code)) {
			// to avoid a repeated insertion of a new person
			return null;
		}

		final DbPerson p = code.startsWith(DNET_PERSON_PREFIX) ? affiliations.get(code) : affiliations.get(masters.get(code));

		if (p == null) {
			log.warn("Person not found, code: " + code);
			dao.registerRawPerson(code, XsltFunctions.capitalize(name), XsltFunctions.capitalize(surname), suffix, orcid);
			newRegistered.add(code);
			return null;
		}

		final CnrPerson cp = new CnrPerson();
		cp.setId(p.getId());
		cp.setName(p.getFullname());
		cp.setOrcid(p.getOrcid());

		return cp;
	}

	protected static boolean isSamePerson(final String fullname, final String other) {

		final String surname = StringUtils.substringBefore(fullname, ",").trim().toLowerCase();
		final String name = StringUtils.substringAfter(fullname, ",").trim().toLowerCase();

		final List<Character> initials1 = Arrays.stream(name.split(" "))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(s -> s.charAt(0))
				.collect(Collectors.toList());

		if (initials1.isEmpty()) { return false; }

		if (!other.toLowerCase().startsWith(surname)) { return false; }

		final String[] parts = StringUtils.substringAfter(other.toLowerCase(), surname)
				.replaceAll("\\,", " ")
				.replaceAll("\\.", " ")
				.split(" ");

		final List<Character> initials2 = Arrays.stream(parts)
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(s -> s.charAt(0))
				.collect(Collectors.toList());

		if (initials2.isEmpty()) {
			return false;
		} else if (initials1.containsAll(initials2)) {
			return true;
		} else if (initials2.containsAll(initials1)) {
			return true;
		} else {
			return false;
		}

	}

}
