package eu.dnetlib.data.mdstore.plugins;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;

import eu.dnetlib.data.mdstore.plugins.objects.CnrCollection;
import eu.dnetlib.data.mdstore.plugins.objects.MdRecord;
import eu.dnetlib.data.utils.HttpFetcher;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.rmi.enabling.ISLookUpService;

public class EnrichCollectionPlugin extends MdRecordPlugin {

	private Map<String, CnrCollection> colls = new HashMap<>();

	private static final Log log = LogFactory.getLog(EnrichCollectionPlugin.class);

	@Autowired
	private UniqueServiceLocator serviceLocator;

	@Override
	protected void reconfigure(final Map<String, String> params) {

		params.entrySet().forEach(e -> log.debug(String.format("******************* %s -> %s", e.getKey(), e.getValue())));

		try {
			final String profile = serviceLocator.getService(ISLookUpService.class).getResourceProfile(params.get("dsId"));
			final Document doc = new SAXReader().read(new StringReader(profile));
			final String baseUrl = doc.valueOf("//INTERFACE[@id='" + params.get("dsInterface") + "']/BASE_URL").trim();

			colls.clear();
			colls.putAll(listOaiCollections(baseUrl, null).stream()
					.map(this::createCollection)
					.filter(Objects::nonNull)
					.distinct()
					.collect(Collectors.toMap(CnrCollection::getCode, o -> o)));

		} catch (final Throwable e) {
			log.error("Error evaluating ListSets", e);
			throw new RuntimeException("Error evaluating ListSets", e);
		}

	}

	@Override
	protected boolean updateRecord(final String recordId, final MdRecord record) {
		log.debug("Updating record: " + recordId);
		for (final CnrCollection c : record.getInCollections()) {
			if (colls.containsKey(c.getCode())) {
				c.setName(colls.get(c.getCode()).getName());
				c.setAcronym(colls.get(c.getCode()).getAcronym());
			}
		}
		log.debug("Record updated: " + recordId);

		return true;
	}

	@SuppressWarnings("unchecked")
	private List<Node> listOaiCollections(final String baseUrl, final String token) {
		final SAXReader reader = new SAXReader();
		try {

			final StringWriter listSetsUrl = new StringWriter();
			listSetsUrl.append(baseUrl);
			listSetsUrl.append("?verb=ListSets");
			if (token != null) {
				listSetsUrl.append("&resumptionToken=");
				listSetsUrl.append(token);

			}
			log.info("ListSets: " + listSetsUrl);
			final String s = HttpFetcher.fetch(listSetsUrl.toString());
			final Document doc = reader.read(new StringReader(s));

			final List<Node> list = doc.selectNodes("//*[local-name() = 'ListSets']/*[local-name() = 'set']");

			final String nextToken = doc.valueOf("//*[local-name() = 'resumptionToken']").trim();
			if (StringUtils.isNotBlank(nextToken)) {
				list.addAll(listOaiCollections(baseUrl, nextToken));
			}
			return list;

		} catch (final DocumentException | URISyntaxException e) {
			log.error("Error listing sets from url: " + baseUrl, e);
			try {
				final Document doc = reader.read(getClass().getResourceAsStream("ListSets.xml"));
				return doc.selectNodes("//*[local-name() = 'ListSets']/*[local-name() = 'set']");
			} catch (final DocumentException e1) {
				log.error("Error listing sets from local file", e1);
				return new ArrayList<>();
			}
		}
	}

	private CnrCollection createCollection(final Node n) {

		final CnrCollection c = new CnrCollection();
		final String code = n.valueOf("./*[local-name() = 'setSpec']").trim();

		c.setCode(n.valueOf("./*[local-name() = 'setSpec']").trim());

		if (code.equalsIgnoreCase("openaire")) {
			c.setAcronym("openaire");
			c.setName("Openaire Collection");
		} else {
			final String desc = n.valueOf("./*[local-name() = 'setDescription']").trim();
			if (StringUtils.isNotBlank(desc)) {
				final Matcher m = Pattern.compile("^Prodotti della ricerca di (.+) \\- (.+)$").matcher(desc);
				if (m.matches()) {
					c.setAcronym(m.group(1));
					c.setName(m.group(2));
				} else {
					c.setAcronym(desc);
					c.setName(desc);
				}
			} else {
				final String name = n.valueOf("./*[local-name() = 'setName']").trim();
				c.setAcronym(name);
				c.setName(name);
			}

		}
		return c;
	}

	@Override
	protected void resetConfiguration() {
		colls.clear();
	}

}
