package eu.dnetlib.data.mapreduce.hbase.dedup;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;

import eu.dnetlib.data.graph.model.DNGFDecoder;
import eu.dnetlib.data.graph.model.DNGFRowKeyDecoder;
import eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;

import com.googlecode.protobuf.format.JsonFormat;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.pace.config.DedupConfig;
import eu.dnetlib.pace.model.gt.GTAuthor;

public class DedupFindRootsPersonMapper extends TableMapper<ImmutableBytesWritable, ImmutableBytesWritable> {

	private DedupConfig dedupConf;

	private ImmutableBytesWritable outKey;

	private ImmutableBytesWritable outValue;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		dedupConf = DedupConfig.load(context.getConfiguration().get(JobParams.DEDUP_CONF));
		System.out.println("dedup findRoots mapper\nwf conf: " + dedupConf.toString());

		outKey = new ImmutableBytesWritable();
		outValue = new ImmutableBytesWritable();
	}

	@Override
	protected void map(final ImmutableBytesWritable rowkey, final Result value, final Context context) throws IOException, InterruptedException {
		// System.out.println("Find root mapping: " + new String(rowkey.copyBytes()));

		final DNGFRowKeyDecoder rkd = DNGFRowKeyDecoder.decode(rowkey.copyBytes());

		if (!Type.person.equals(rkd.getType())) {
			context.getCounter(rkd.getType().toString(), "skipped").increment(1);
		}

		final Type type = Type.valueOf(dedupConf.getWf().getEntityType());
		final Map<byte[], byte[]> similarRels = value.getFamilyMap(HBaseTableDAO.getSimilarityQualifierBytes(type));

		if ((similarRels != null) && !similarRels.isEmpty()) {
			final ByteBuffer min = findMin(rowkey.copyBytes(), similarRels.keySet());

			final byte[] groupingKey = HBaseTableDAO.newIdBytes(min, dedupConf.getWf().getDedupRun());

			final GTAuthor gta = asGTA(rowkey, value.getValue(Bytes.toBytes(dedupConf.getWf().getEntityType()), HBaseTableDAO.cfMetadataByte()));

			emitBody(context, groupingKey, gta);
		} else {
			context.getCounter(dedupConf.getWf().getEntityType(), "row not in similarity mesh").increment(1);
		}
	}

	private GTAuthor asGTA(final ImmutableBytesWritable rowkey, final byte[] input) {

		final DNGFDecoder decoder = DNGFDecoder.decode(input);
		final DNGF oaf = decoder.getDNGF();

		final Person person = oaf.getEntity().getPerson();

		final GTAuthor gta = GTAuthor.fromDNGFJson(new JsonFormat().printToString(person));
		final String id = new String(rowkey.copyBytes());
		gta.setId(id);
		gta.getAuthor().setId(id);
		return gta;
	}

	private ByteBuffer findMin(final byte[] key, final Iterable<byte[]> keys) {
		ByteBuffer bb = ByteBuffer.wrap(key);
		for (final byte[] q : keys) {
			final ByteBuffer iq = ByteBuffer.wrap(q);
			if (bb.compareTo(iq) > 0) {
				bb = iq;
			}
		}
		return bb;
	}

	private void emitBody(final Context context, final byte[] row, final GTAuthor gta) throws IOException, InterruptedException {

		outKey.set(row);
		outValue.set(toDNGFByteArray(gta));

		context.write(outKey, outValue);
		context.getCounter(dedupConf.getWf().getEntityType(), "in").increment(1);
	}

	public byte[] toDNGFByteArray(final GTAuthor gta) {
		// final DNGF oaf = new GTAuthorMapper().map(gta);
		// return oaf.toByteArray();
		return Bytes.toBytes(gta.toString());
	}

}
