package eu.dnetlib.data.mapreduce.hbase.dli;

import java.io.IOException;

import eu.dnetlib.data.graph.model.DNGFRowKeyDecoder;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.hbase.dli.kv.DliKey;
import eu.dnetlib.data.proto.DNGFProtos.DNGF;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.transform.Ontologies;
import eu.dnetlib.data.transform.OntologyLoader;
import eu.dnetlib.dli.proto.ScholixConverter;
import eu.dnetlib.dli.proto.ScholixVersion;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;

import static eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO.getMetadata;
import static eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO.rel;

/**
 * Created by sandro on 2/15/17.
 */
public class PrepareScholixDataMapper extends TableMapper<DliKey, ImmutableBytesWritable> {

    private final static String SCHOLIX = "Scholix";
    private final static String DNGF = "DNGF";
    private String scholixVersion ;
    private Ontologies ontologies;


    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        ontologies = OntologyLoader.loadOntologies(context.getConfiguration().get(JobParams.ONTOLOGIES));
        scholixVersion = context.getConfiguration().get(JobParams.SCHOLIXVERSION);

        System.out.println("ontologies: " + ontologies.toJson(true));
        System.out.println("scholixVersion = " + scholixVersion);
    }

    @Override
    protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

        final Type type = DNGFRowKeyDecoder.decode(key.copyBytes()).getType();
        final DNGF entity = getMetadata(value, type);

        if (isValid(entity)) {
            if (!deletedByInference(entity)) {

                final byte[] source = ScholixConverter.withSource(entity.getEntity(), ScholixVersion.valueOf(scholixVersion));
                emit(DliKey.ENTITY, entity.getEntity().getId(), context, source);
                context.getCounter(SCHOLIX, "Source: " + entity.getEntity().getType().name()).increment(1);

                rel(value, "isMergedIn", "merges", "isSimilarTo").values().forEach(r -> {
                    if (!deletedByInference(r)) {
                        final byte [] target =  ScholixConverter.withTarget(entity.getEntity(), r.getRel(), ontologies, ScholixVersion.valueOf(scholixVersion));
                        emit(DliKey.REL, r.getRel().getTarget(), context, target);
                        context.getCounter(SCHOLIX, "Rel type: " + r.getRel().getTargetType().name()).increment(1);
                    } else {
                        context.getCounter(DNGF, "Rel deletedbyinference: " + type.name()).increment(1);
                    }
                });
            } else {
                context.getCounter(DNGF, "Deletedbyinference: " + type.name()).increment(1);
            }
        } else {
            context.getCounter(DNGF, "Invalid: " + type.name()).increment(1);
        }
    }

    private boolean isValid(final DNGF entity) {
        return (entity != null) && entity.isInitialized();
    }

    private boolean deletedByInference(final DNGF oaf) {
        return oaf.getDataInfo().getDeletedbyinference();
    }

    private void emit(final int keyType, final String id, final Context context, final byte[] data) {
        try {
            context.getCounter(SCHOLIX, keyType == DliKey.ENTITY ? "ENTITY" : "REL").increment(1);
            context.write(DliKey.create(keyType, id), new ImmutableBytesWritable(data));
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }
}
