package eu.dnetlib.data.mapreduce.hbase.dataimport;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.google.protobuf.Descriptors;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO;
import eu.dnetlib.data.proto.DNGFProtos;
import eu.dnetlib.data.proto.KindProtos;
import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformer;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;

import static eu.dnetlib.data.mapreduce.util.dao.HBaseTableDAO.*;

public class ImportRecordsMapper extends Mapper<Text, Text, ImmutableBytesWritable, Put> {

    /**
     * logger.
     */
    private static final Log log = LogFactory.getLog(ImportRecordsMapper.class); // NOPMD by marko on 11/24/08 5:02 PM

    private XsltRowTransformer transformer;

    private ImmutableBytesWritable ibw;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {
        super.setup(context);

        final String xslt = context.getConfiguration().get(JobParams.HBASE_IMPORT_XSLT).trim();

        if ((xslt == null) || xslt.isEmpty()) throw new IllegalArgumentException("missing xslt");
        final Map<String, Object> xslParams = Maps.newHashMap();

        if (context.getConfiguration().get("datasourceTypeMap") != null) {
            final Map<String, String> dsMap = getDatasourceTypeMap(context);
            log.info("using datasource type map:\n" + dsMap.toString());
            xslParams.put("mergeIdForHomonymsMap", dsMap);
        }

        transformer = XsltRowTransformerFactory.newInstance(xslt, xslParams);

        ibw = new ImmutableBytesWritable();

        log.info("got xslt: '" + xslt);
        log.info("using transformer: '" + transformer.getTransformerClassName() + "'");
    }

    @Override
    protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {
        try {
            transformer.apply(value.toString()).forEach(row -> {
                final byte[] rowKey = Bytes.toBytes(row.getKey());
                ibw.set(rowKey);
                row.forEach(col -> {
                    try {
                        final DNGFProtos.DNGF dngf = parseProto(col.getValue());
                        context.write(ibw, asPutByCollectedFrom(dngf));
                        context.getCounter("mdstore", row.getColumnFamily()).increment(row.getColumns().size());
                        final String rkey = new String(rowKey).split("::")[0];
                        if (dngf.getKind() == KindProtos.Kind.entity) {
                            context.getCounter("MetadataCollectedFromNSPrefix", rkey).increment(1);
                        } else {
                            context.getCounter("RelationCollectedFromNSPrefix", rkey).increment(1);
                        }
                    } catch (Exception e) {
                        log.error("error importing the following record on HBase: " + value.toString(), e);
                        context.getCounter("MetadataError", "xslt Error").increment(1);
                    }
                });
            });
        } catch (Throwable e) {
            context.getCounter("MetadataError", "xslt Error").increment(1);
        }
    }

    private Map<String, String> getDatasourceTypeMap(final Context context) {
        final String dsType = context.getConfiguration().get("datasourceTypeMap");

        final Type token = new TypeToken<Map<String, String>>() {
        }.getType();
        final Map<String, String> dsMap = new Gson().fromJson(dsType, token);
        return dsMap;
    }

}
