package eu.dnetlib.data.mapreduce.hbase.broker;

import com.google.gson.Gson;
import eu.dnetlib.data.graph.utils.HBaseTableUtils;
import eu.dnetlib.data.mapreduce.hbase.broker.model.ESEvent;
import eu.dnetlib.data.mapreduce.hbase.broker.model.OpenAireEventPayload;
import eu.dnetlib.data.mapreduce.hbase.broker.model.Publication;
import eu.dnetlib.data.mapreduce.hbase.dli.kv.DliKey;
import eu.dnetlib.data.mapreduce.hbase.broker.model.Pid;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


public class DatasetRelationEnrichmentReducer extends Reducer<DliKey, Text, Text, Text> {

    private Text outValue;

    private Text outKey;
    private Gson gson;

    @Override
    protected void setup(Context context) throws IOException, InterruptedException {
        super.setup(context);
        outValue = new Text();
        outKey = new Text();
        gson = new Gson();
    }

    @Override
    protected void reduce(DliKey key, Iterable<Text> values, Context context) throws IOException, InterruptedException {
        Iterator<Text> v_iter = values.iterator();
        Text first = v_iter.next();
        Publication targetPublication;
        try {
            targetPublication = gson.fromJson(first.toString(), Publication.class);
        } catch (Throwable e) {
            context.getCounter("Broker Enrichment", "Error first element unexpected: ").increment(1);
            return;
        }


        if (!isValidTarget(targetPublication)) {
            context.getCounter("Broker Enrichment", "Error first element missing target ").increment(1);
            return;
        }

        if (!v_iter.hasNext()) {
            context.getCounter("Broker Enrichment", "NOOP").increment(1);
            return;
        }


        Map<String, ESEvent> events = new HashMap<>();

        while (v_iter.hasNext()) {

            final ESEvent currentEvent = gson.fromJson(v_iter.next().toString(), ESEvent.class);

            final OpenAireEventPayload currentPayload  =gson.fromJson(currentEvent.getPayload(), OpenAireEventPayload.class);
            currentEvent.getMap().setTarget_publication_title(currentPayload.getPublication().getTitles().get(0));
            String eventIdentifier;

            if (targetPublication.getDatasets()!= null && targetPublication.getDatasets().size() >0 ) {
                eventIdentifier=  generateEventIdentifier(currentPayload.getPublication().getPids().get(0), targetPublication.getDatasets().get(0).getPids().get(0),currentEvent.getTopic());
                currentPayload.getPublication().setDatasets(targetPublication.getDatasets());
            } else {
                eventIdentifier=  generateEventIdentifier(currentPayload.getPublication().getPids().get(0), targetPublication.getPublications().get(0).getPids().get(0),currentEvent.getTopic());

                currentPayload.getPublication().setPublications(targetPublication.getPublications());
            }
            currentEvent.setEventId(eventIdentifier);
            currentPayload.setHighlight(targetPublication);
            currentEvent.setPayload(gson.toJson(currentPayload));
            events.put(currentEvent.getEventId(), currentEvent);
        }
        events.values().forEach(r -> {
                emit(context, gson.toJson(r));
                context.getCounter("Broker Enrichment Event", "Generated Event").increment(1);
                });
    }


    private String generateEventIdentifier(final Pid source, final Pid target, final String topic) {
        return calculateEventId(topic, String.format("%s:%s",source.getType(), source.getValue()),String.format("%s:%s",target.getType(), target.getValue()));
    }


    private String calculateEventId(final String topic, final String sourceId, final String targetId) {
        return "event-"
                + DigestUtils.md5Hex(topic).substring(0, 6) + "-"
                + DigestUtils.md5Hex(sourceId).substring(0, 8) + "-"
                + DigestUtils.md5Hex(targetId).substring(0, 8);
    }




    private boolean isValidTarget(final Publication p ) {
        return ((p.getPublications() != null && p.getPublications().size()>0) || (p.getDatasets() !=null && p.getDatasets().size() >0));

    }



    private void emit(final Context context, final String data) {
        outValue.set(data.getBytes());
        try {
            context.write(outKey, outValue);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }
}
