package eu.dnetlib.data.mapreduce.util;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;
import com.googlecode.protobuf.format.XmlFormat;
import com.mycila.xmltool.XMLDoc;
import com.mycila.xmltool.XMLTag;
import eu.dnetlib.data.graph.model.DNGFDecoder;
import eu.dnetlib.data.graph.model.DNGFEntityDecoder;
import eu.dnetlib.data.graph.model.DNGFRelDecoder;
import eu.dnetlib.data.graph.utils.RelDescriptor;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextDef;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.proto.DNGFProtos.DNGFEntity;
import eu.dnetlib.data.proto.DNGFProtos.DNGFRel;
import eu.dnetlib.data.proto.FieldTypeProtos.*;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.ProjectProtos.Project;
import eu.dnetlib.data.proto.PublicationProtos.Publication;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.proto.WdsDatasetProtos.WdsDataset.GeoLocation;
import eu.dnetlib.data.transform.Ontologies;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

public class XmlRecordFactory extends AbstractRecordFactory {

    protected TemplateFactory templateFactory = new TemplateFactory();

    protected List<String> extraInfo = Lists.newArrayList();

    protected Transformer transformer;
    private  final String schemaLocation;


    public XmlRecordFactory(final EntityConfigTable entityConfigTable, final ContextMapper contextMapper, final Ontologies ontologies,
                            final String schemaLocation, final boolean entityDefaults, final boolean relDefaults, final boolean childDefeaults)
            throws TransformerConfigurationException, TransformerFactoryConfigurationError {
        super(entityConfigTable, contextMapper, ontologies,  entityDefaults, relDefaults, childDefeaults);
        this.schemaLocation = schemaLocation;
        transformer = TransformerFactory.newInstance().newTransformer();
        transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
    }

    public static String escapeXml(final String value) {
        return value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll("\"", "&quot;").replaceAll("'", "&apos;");
    }

    @Override
    public String build() {
        try {
            final DNGFEntityDecoder entity = mainEntity.decodeEntity();
            // log.info("building");
            // log.info("main: " + mainEntity);
            // log.info("rel:  " + relations);
            // log.info("chi:  " + children);
            // log.info("=============");

            final Type type = entity.getType();
            final List<String> metadata = decodeType(entity, null, entityDefaults, false);

            // rels has to be processed before the contexts because they enrich the contextMap with the funding info.
            final List<String> rels = listRelations();
            metadata.addAll(buildContexts(type));
            metadata.add(parseDataInfo(mainEntity));

            final String body = templateFactory.buildBody(type, metadata, rels, listChildren(), extraInfo);

            return templateFactory
                    .buildRecord(type, key, entity.getDateOfCollection(), entity.getDateOfTransformation(), schemaLocation, body, countersAsXml());
        } catch (final Throwable e) {
            throw new RuntimeException(String.format("error building record '%s'", this.key), e);
        }
    }

    private String parseDataInfo(final DNGFDecoder decoder) {
        final DataInfo dataInfo = decoder.getDNGF().getDataInfo();

        final StringBuilder sb = new StringBuilder();
        sb.append("<datainfo>");
        sb.append(asXmlElement("inferred", dataInfo.getInferred() + "", null, null));
        sb.append(asXmlElement("deletedbyinference", dataInfo.getDeletedbyinference() + "", null, null));
        sb.append(asXmlElement("trust", dataInfo.getTrust() + "", null, null));
        sb.append(asXmlElement("inferenceprovenance", dataInfo.getInferenceprovenance() + "", null, null));
        sb.append(asXmlElement("provenanceaction", null, dataInfo.getProvenanceaction(), null));
        sb.append("</datainfo>");

        return sb.toString();
    }

    private List<String> decodeType(final DNGFEntityDecoder decoder, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

        final List<String> metadata = Lists.newArrayList();
        metadata.addAll(listFields(decoder.getMetadata(), filter, defaults, expandingRel));
        metadata.addAll(listFields(decoder.getDNGFEntity(), filter, defaults, expandingRel));

        if ((decoder.getEntity() instanceof Publication) && !expandingRel) {
            metadata.add(asXmlElement("bestlicense", "", getBestLicense(), null));

            metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
        }
        if ((decoder.getEntity() instanceof Person) && !expandingRel) {
            metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
        }
        if ((decoder.getEntity() instanceof Project) && !expandingRel) {
            metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
        }

        return metadata;
    }


    private List<String> listRelations() {

        final List<String> rels = Lists.newArrayList();

        for (final DNGFDecoder decoder : this.relations) {

            final DNGFRel rel = decoder.getDNGFRel();
            final DNGFEntity cachedTarget = rel.getCachedTarget();
            final DNGFRelDecoder relDecoder = DNGFRelDecoder.decode(rel);

            // if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(key)) {
            if (relDecoder.getRelSourceId().equals(key) || relDecoder.getRelTargetId().equals(key)) {

                final List<String> metadata = Lists.newArrayList();
                final Type targetType = relDecoder.getTargetType();
                final Set<String> relFilter = entityConfigTable.getFilter(targetType, relDecoder.getRelDescriptor());
                metadata.addAll(listFields(relDecoder.getDngfRel(), relFilter, false, true));


                final RelDescriptor relDescriptor = relDecoder.getRelDescriptor();

                if ((cachedTarget != null) && cachedTarget.isInitialized()) {

                    final Set<String> filter = entityConfigTable.getFilter(targetType, relDescriptor);
                    metadata.addAll(decodeType(DNGFEntityDecoder.decode(cachedTarget), filter, relDefaults, true));
                }

                final String semanticclass = ontologies.inverseOf(relDescriptor);
                final String semanticscheme = relDescriptor.getOntologyCode();

                final String rd = semanticscheme + "_" + semanticclass;
                incrementCounter(rd);

                final DataInfo info = decoder.getDNGF().getDataInfo();
                manageInferred(rd, info);

                final LinkDescriptor ld = entityConfigTable.getDescriptor(relDecoder.getTargetType(), relDescriptor);

                final String relId = (ld != null) && !ld.isSymmetric() ? relDecoder.getRelTargetId() : relDecoder.getRelSourceId();

                rels.add(templateFactory.getRel(targetType, relId, metadata, semanticclass, semanticscheme, info.getInferred(), info.getTrust(),
                        info.getInferenceprovenance(), info.getProvenanceaction().getClassid()));
            }
        }
        return rels;
    }



    // //////////////////////////////////

    private List<String> listChildren() {

        final List<String> children = Lists.newArrayList();
        for (final DNGFDecoder decoder : this.children) {
            final DNGFEntity cachedTarget = decoder.getDNGFRel().getCachedTarget();
            addChildren(children, cachedTarget, decoder.getRelDescriptor());
        }
        final DNGFEntityDecoder entity = mainEntity.decodeEntity();
        if (entity.getType().equals(Type.publication)) {
            for (final Instance instance : ((Publication) entity.getEntity()).getInstanceList()) {
                final Set<String> instanceFieldFilter = Sets.newHashSet("instancetype", "hostedby", "licence");
                final Function<String, String> wrExtractor = identifier -> templateFactory.getWebResource(identifier);
                final List<String> webResources = Lists.newArrayList(Iterables.transform(instance.getUrlList(), wrExtractor));
                final List<String> instanceFields = listFields(instance, instanceFieldFilter, false, false);
                children.add(templateFactory.getInstance(instance.getHostedby().getKey(), instanceFields, webResources));
            }
            for (final ExternalReference er : ((Publication) entity.getEntity()).getExternalReferenceList()) {
                // Set<String> filters = entityConfigTable.getFilter(Type.result, RelType.resultResult);
                final List<String> fields = listFields(er, null, false, false);
                children.add(templateFactory.getChild("externalreference", null, fields));
            }
        }

        return children;
    }

    private void addChildren(final List<String> children, final DNGFEntity target, final RelDescriptor relDescriptor) {
        final DNGFEntityDecoder decoder = DNGFEntityDecoder.decode(target);
        incrementCounter(relDescriptor.getTermCode());
        final Set<String> filters = entityConfigTable.getFilter(target.getType(), relDescriptor);
        children.add(templateFactory.getChild(decoder.getType().toString(), decoder.getId(), listFields(decoder.getMetadata(), filters, childDefaults, false)));
    }

    private List<String> listFields(final GeneratedMessage fields, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

        final List<String> metadata = Lists.newArrayList();

        if (fields != null) {

            final Set<String> seen = Sets.newHashSet();
            for (final Entry<FieldDescriptor, Object> e : filterFields(fields, filter)) {

                // final String name = getFieldName(e.getKey().getName());
                final String name = e.getKey().getName();
                seen.add(name);

                addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
            }

            if (defaults) {
                for (final FieldDescriptor fd : fields.getDescriptorForType().getFields()) {
                    if (!seen.contains(fd.getName())) {
                        addFieldValue(metadata, fd, getDefault(fd), expandingRel);
                    }
                }
            }
        }
        return metadata;
    }





    @SuppressWarnings("unchecked")
    private void addFieldValue(final List<String> metadata, final FieldDescriptor fd, final Object value, final boolean expandingRel) {
        if ("dateofcollection".equals(fd.getName()) ||
                "dateoftransformation".equals(fd.getName()) ||
                "id".equals(fd.getName()) ||
                (value == null)) return;

        if (fd.getName().equals("datasourcetype")) {
            final String classid = ((Qualifier) value).getClassid();

            final Qualifier.Builder q = Qualifier.newBuilder((Qualifier) value);
            if (specialDatasourceTypes.contains(classid)) {
                q.setClassid("other").setClassname("other");
            }
            metadata.add(asXmlElement("datasourcetypeui", "", q.build(), null));
        }

        if (fd.isRepeated() && (value instanceof List<?>)) {
            for (final Object o : (List<Object>) value) {
                guessType(metadata, fd, o, expandingRel);
            }
        } else {
            guessType(metadata, fd, value, expandingRel);
        }
    }

    private void guessType(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {

        if (fd.getType().equals(FieldDescriptor.Type.MESSAGE)) {

            if (Qualifier.getDescriptor().equals(fd.getMessageType())) {
                final Qualifier qualifier = (Qualifier) o;
                metadata.add(asXmlElement(fd.getName(), "", qualifier, null));
            }

            if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
                final StructuredProperty sp = (StructuredProperty) o;
                metadata.add(asXmlElement(fd.getName(), sp.getValue(), sp.getQualifier(), sp.hasDataInfo() ? sp.getDataInfo() : null));

                if (!expandingRel && fd.getName().equals("pid")) {
                    if (sp.getQualifier().getClassid().equalsIgnoreCase("doi")) {
                        incrementCounter("doi");
                    }
                }
            }

            if (KeyValue.getDescriptor().equals(fd.getMessageType())) {
                final KeyValue kv = (KeyValue) o;
                metadata.add("<" + fd.getName() + " name=\"" + escapeXml(kv.getValue()) + "\" id=\"" + escapeXml(removePrefix(kv.getKey())) + "\"/>");
            }

            if (StringField.getDescriptor().equals(fd.getMessageType())) {
                final String fieldName = fd.getName();

                if (fieldName.equals("fundingtree")) {
                    final String xmlTree = o instanceof StringField ? ((StringField) o).getValue() : o.toString();

                    if (expandingRel) {
                        metadata.add(getRelFundingTree(xmlTree));
                        fillContextMap(xmlTree);
                    } else {
                        metadata.add(xmlTree);
                    }
                } else {
                    final StringField sf = (StringField) o;
                    final StringBuilder sb = new StringBuilder("<" + fd.getName());
                    if (sf.hasDataInfo()) {
                        final DataInfo dataInfo = sf.getDataInfo();
                        dataInfoAsAttributes(sb, dataInfo);
                    }
                    sb.append(">" + escapeXml(sf.getValue()) + "</" + fd.getName() + ">");
                    metadata.add(sb.toString());
                }
            }

            if (BoolField.getDescriptor().equals(fd.getMessageType())) {
                final BoolField bf = (BoolField) o;
                final StringBuilder sb = new StringBuilder("<" + fd.getName());
                if (bf.hasDataInfo()) {
                    final DataInfo dataInfo = bf.getDataInfo();
                    dataInfoAsAttributes(sb, dataInfo);
                }

                sb.append(">" + (bf.hasValue() ? bf.getValue() : "") + "</" + fd.getName() + ">");
                metadata.add(sb.toString());
            }

            if (Journal.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
                final Journal j = (Journal) o;
                metadata.add("<journal " + "issn=\"" + escapeXml(j.getIssnPrinted()) + "\" " + "eissn=\"" + escapeXml(j.getIssnOnline()) + "\" " + "lissn=\""
                        + escapeXml(j.getIssnLinking()) + "\">" + escapeXml(j.getName()) + "</journal>");
            }

            if (Context.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
                contextes.add(((Context) o).getId());
            }

            if (ExtraInfo.getDescriptor().equals(fd.getMessageType()) && (o != null)) {

                final ExtraInfo e = (ExtraInfo) o;
                final StringBuilder sb = new StringBuilder("<" + fd.getName() + " ");

                sb.append("name=\"" + e.getName() + "\" ");
                sb.append("typology=\"" + e.getTypology() + "\" ");
                sb.append("provenance=\"" + e.getProvenance() + "\" ");
                sb.append("trust=\"" + e.getTrust() + "\"");
                sb.append(">");
                sb.append(e.getValue());
                sb.append("</" + fd.getName() + ">");

                extraInfo.add(sb.toString());
            }

            if (GeoLocation.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
                final String geoLocation = XmlFormat.printToString((GeoLocation) o);
                metadata.add(geoLocation);
            }

        } else if (fd.getType().equals(FieldDescriptor.Type.ENUM)) {
            if (fd.getFullName().equals("eu.dnetlib.data.proto.DNGFEntity.type")) return;
            metadata.add(asXmlElement(fd.getName(), ((EnumValueDescriptor) o).getName(), null, null));
        } else {
            metadata.add(asXmlElement(fd.getName(), o.toString(), null, null));
        }
    }

    private StringBuilder dataInfoAsAttributes(final StringBuilder sb, final DataInfo dataInfo) {
        sb.append(" inferred=\"" + dataInfo.getInferred() + "\"");
        sb.append(" inferenceprovenance=\"" + dataInfo.getInferenceprovenance() + "\"");
        sb.append(" provenanceaction=\"" + dataInfo.getProvenanceaction().getClassid() + "\"");
        sb.append(" trust=\"" + dataInfo.getTrust() + "\" ");
        return sb;
    }

    private List<String> buildContexts(final Type type) {
        final List<String> res = Lists.newArrayList();

        if ((contextMapper != null) && !contextMapper.isEmpty() && type.equals(Type.publication)) {

            XMLTag document = XMLDoc.newDocument(true).addRoot("contextRoot");

            for (final String context : contextes) {

                String id = "";
                for (final String token : Splitter.on("::").split(context)) {
                    id += token;

                    final ContextDef def = contextMapper.get(id);

                    if (def == null)
                        throw new IllegalStateException(String.format("cannot find context for id '%s'", id));

                    if (def.getName().equals("context")) {
                        final String xpath = "//context/@id='" + def.getId() + "'";
                        if (!document.gotoRoot().rawXpathBoolean(xpath, new Object())) {
                            document = addContextDef(document.gotoRoot(), def);
                        }
                    }

                    if (def.getName().equals("category")) {
                        final String rootId = StringUtils.substringBefore(def.getId(), "::");
                        document = addContextDef(document.gotoRoot().gotoTag("//context[./@id='" + rootId + "']", new Object()), def);
                    }

                    if (def.getName().equals("concept")) {
                        document = addContextDef(document, def).gotoParent();
                    }
                    id += "::";
                }
            }

            for (final org.w3c.dom.Element x : document.gotoRoot().getChildElement()) {
                try {
                    res.add(asStringElement(x));
                } catch (final TransformerException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        return res;
    }

    private XMLTag addContextDef(final XMLTag tag, final ContextDef def) {
        tag.addTag(def.getName()).addAttribute("id", def.getId()).addAttribute("label", def.getLabel());
        if ((def.getType() != null) && !def.getType().isEmpty()) {
            tag.addAttribute("type", def.getType());
        }
        return tag;
    }

    private String asStringElement(final org.w3c.dom.Element element) throws TransformerException {
        final StringWriter buffer = new StringWriter();
        transformer.transform(new DOMSource(element), new StreamResult(buffer));
        return buffer.toString();
    }

    @SuppressWarnings("unchecked")
    private String getRelFundingTree(final String xmlTree) {
        String funding = "<funding>";
        try {
            final Document ftree = new SAXReader().read(new StringReader(xmlTree));
            funding = "<funding>";
            // String _id = "";

            funding += getFunderElement(ftree);

            for (final Object o : Lists.reverse(ftree.selectNodes("//fundingtree//*[starts-with(local-name(),'funding_level_')]"))) {
                final Element e = (Element) o;
                final String _id = e.valueOf("./id");
                funding += "<" + e.getName() + " name=\"" + escapeXml(e.valueOf("./name")) + "\">" + escapeXml(_id) + "</" + e.getName() + ">";
                // _id += "::";
            }
        } catch (final DocumentException e) {
            throw new IllegalArgumentException("unable to parse funding tree: " + xmlTree + "\n" + e.getMessage());
        } finally {
            funding += "</funding>";
        }
        return funding;
    }

    private String getFunderElement(final Document ftree) {
        final String funderId = ftree.valueOf("//fundingtree/funder/id/text()");
        final String funderShortName = ftree.valueOf("//fundingtree/funder/shortname/text()");
        final String funderName = ftree.valueOf("//fundingtree/funder/name/text()");
        final String funderJurisdiction = ftree.valueOf("//fundingtree/funder/jurisdiction/text()");

        return "<funder id=\"" + escapeXml(funderId) + "\" shortname=\"" + escapeXml(funderShortName) + "\" name=\"" + escapeXml(funderName)
                + "\" jurisdiction=\"" + escapeXml(funderJurisdiction) + "\" />";
    }

    private void fillContextMap(final String xmlTree) {

        Document fundingPath;
        try {
            fundingPath = new SAXReader().read(new StringReader(xmlTree));
        } catch (final DocumentException e) {
            throw new RuntimeException(e);
        }
        try {
            final Node funder = fundingPath.selectSingleNode("//funder");

            if (funder != null) {

                final String funderShortName = funder.valueOf("./shortname");
                contextes.add(funderShortName);

                contextMapper.put(funderShortName, new ContextDef(funderShortName, funder.valueOf("./name"), "context", "funding"));
                final Node level0 = fundingPath.selectSingleNode("//funding_level_0");
                if (level0 != null) {
                    final String level0Id = Joiner.on("::").join(funderShortName, level0.valueOf("./name"));
                    contextMapper.put(level0Id, new ContextDef(level0Id, level0.valueOf("./description"), "category", ""));
                    final Node level1 = fundingPath.selectSingleNode("//funding_level_1");
                    if (level1 == null) {
                        contextes.add(level0Id);
                    } else {
                        final String level1Id = Joiner.on("::").join(level0Id, level1.valueOf("./name"));
                        contextMapper.put(level1Id, new ContextDef(level1Id, level1.valueOf("./description"), "concept", ""));
                        final Node level2 = fundingPath.selectSingleNode("//funding_level_2");
                        if (level2 == null) {
                            contextes.add(level1Id);
                        } else {
                            final String level2Id = Joiner.on("::").join(level1Id, level2.valueOf("./name"));
                            contextMapper.put(level2Id, new ContextDef(level2Id, level2.valueOf("./description"), "concept", ""));
                            contextes.add(level2Id);
                        }
                    }
                }
            }
        } catch (final NullPointerException e) {
            throw new IllegalArgumentException("malformed funding path: " + xmlTree, e);
        }
    }

    private String asXmlElement(final String name, final String value, final Qualifier q, final DataInfo dataInfo) {
        StringBuilder sb = new StringBuilder();
        sb.append("<");
        sb.append(name);
        if (q != null) {
            sb.append(getAttributes(q));
        }
        if (dataInfo != null) {
            sb = dataInfoAsAttributes(sb, dataInfo);
        }
        if ((value == null) || value.isEmpty()) {
            sb.append("/>");
            return sb.toString();
            // return "<" + name + getAttributes(q) + "/>";
        }

        sb.append(">");
        // sb.append(escapeXml(Normalizer.normalize(value, Normalizer.Form.NFD)));
        sb.append(escapeXml(value));
        sb.append("</");
        sb.append(name);
        sb.append(">");

        return sb.toString();
        // return "<" + name + getAttributes(q) + ">" + escapeXml(value) + "</" + name + ">";
    }

    private String getAttributes(final Qualifier q) {
        if (q == null) return "";

        final StringBuilder sb = new StringBuilder();
        for (final Entry<FieldDescriptor, Object> e : q.getAllFields().entrySet()) {
            // sb.append(" " + e.getKey().getName() + "=\"" + escapeXml(e.getValue().toString()) + "\"");
            sb.append(" ");
            sb.append(e.getKey().getName());
            sb.append("=\"");
            sb.append(escapeXml(e.getValue().toString()));
            sb.append("\"");
        }
        return sb.toString();
    }


    private List<String> countersAsXml() {
        final List<String> out = Lists.newArrayList();
        for (final Entry<String, Integer> e : counters.entrySet()) {
            out.add(String.format("<counter_%s value=\"%s\"/>", e.getKey(), e.getValue()));
        }
        return out;
    }



    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("################################################\n");
        sb.append("ID: ").append(key).append("\n");
        if (mainEntity != null) {
            sb.append("MAIN ENTITY:\n").append(mainEntity.getEntity().toString() + "\n");
        }
        if (relations != null) {
            sb.append("\nRELATIONS:\n");
            for (final DNGFDecoder decoder : relations) {
                sb.append(decoder.getDNGFRel().toString() + "\n");
            }
        }
        if (children != null) {
            sb.append("\nCHILDREN:\n");
            for (final DNGFDecoder decoder : children) {
                sb.append(decoder.getDNGFRel().toString() + "\n");
            }
        }
        return sb.toString();
    }

}
