package eu.dnetlib.data.mapreduce.hbase.broker.model;

import eu.dnetlib.data.proto.DNGFProtos;
import eu.dnetlib.data.proto.DliProtos;
import eu.dnetlib.data.proto.FieldTypeProtos;
import eu.dnetlib.data.proto.TypeProtos;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class Dataset {

    private String originalId;

    private List<String> titles;

    private List<String> collectedFrom;

    private List<Pid> pids;

    public Dataset() {}


    public static Dataset DatasetFromDNGF(DNGFProtos.DNGFEntity mainEntity) {
        if(mainEntity.getType()!= TypeProtos.Type.dataset) {
            throw new RuntimeException("EXPECTED Dataset");
        }


        final Dataset d = new Dataset();
        d.setTitles(mainEntity.getDataset().getMetadata().getTitleList().stream().map(FieldTypeProtos.StructuredProperty::getValue).collect(Collectors.toList()));
        d.setPids(Collections.singletonList(mainEntity.getExtension(DliProtos.typedIdentifier)
                .stream()
                .map(pid ->
                        new Pid()
                                .setType(pid.getQualifier().getClassid())
                                .setValue(pid.getValue()))
                                .filter(distinctByKey(p-> String.format("%s::%s", p.getType().toLowerCase(),p.getValue().toLowerCase())))
                .findFirst().get()));
        d.setOriginalId(d.getPids().get(0).getValue());
        return d;
    }

    public static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor)
    {
        Map<Object, Boolean> map = new ConcurrentHashMap<>();
        return t -> map.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }



    public Dataset(final List<String> titles, final List<String> collectedFrom, final List<Pid> pids) {
        this.titles = titles;
        this.collectedFrom = collectedFrom;
        this.pids = pids;
    }

    public List<String> getTitles() {
        return this.titles;
    }

    public Dataset setTitles(final List<String> titles) {
        this.titles = titles;
        return this;
    }

    public List<String> getCollectedFrom() {
        return this.collectedFrom;
    }

    public Dataset setCollectedFrom(final List<String> collectedFrom) {
        this.collectedFrom = collectedFrom;
        return this;
    }

    public List<Pid> getPids() {
        return this.pids;
    }

    public Dataset setPids(final List<Pid> pids) {
        this.pids = pids;
        return this;
    }

    public String getOriginalId() {
        return originalId;
    }

    public void setOriginalId(String originalId) {
        this.originalId = originalId;
    }
}